import { ReferenceType } from '../../src/types/reference_type';
import { Value } from '../../src/types/value';
import { compileWithAggregate, jsonify, transformAggregatedData } from './test_helpers';

describe('Transform Aggregated Data Tests', () => {
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  function mockDataMergeRequests(): Record<string, any> {
    return {
      project: {
        glql_s1: { count: 1876, totalTimeToMerge: 1194422204.332011 },
        glql_s2: { count: 2650, totalTimeToMerge: 1761566797.487367 },
        glql_s3: { count: 3103, totalTimeToMerge: 2086576128.855193 },
        glql_s4: { count: 2542, totalTimeToMerge: 1662207119.005869 },
        glql_s5: { count: 2037, totalTimeToMerge: 1380841068.758804 },
        glql_s6: { count: 2572, totalTimeToMerge: 2251245834.710288 },
        glql_s7: { count: 2716, totalTimeToMerge: 1946223159.426047 },
        glql_s8: { count: 2894, totalTimeToMerge: 2020946011.821066 },
        glql_s9: { count: 2993, totalTimeToMerge: 2021312509.793126 },
        glql_s10: { count: 2649, totalTimeToMerge: 1834820303.043201 },
        glql_s11: { count: 2762, totalTimeToMerge: 2034516437.573138 },
        glql_s12: { count: 3114, totalTimeToMerge: 1926510508.380342 },
        glql_s13: { count: 667, totalTimeToMerge: 451269078.168328 },
      },
    };
  }

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  function mockDataMergeRequestsTwoDimensions(): Record<string, any> {
    return {
      project: {
        // Time segment 1 (2024-08-09 to 2024-10-01) with 3 users
        glql_s1_s6: { count: 125, totalTimeToMerge: 85000000 }, // alice
        glql_s1_s7: { count: 98, totalTimeToMerge: 72000000 }, // bob
        glql_s1_s8: { count: 156, totalTimeToMerge: 110000000 }, // charlie

        // Time segment 2 (2024-10-01 to 2025-01-01) with 3 users
        glql_s2_s6: { count: 180, totalTimeToMerge: 125000000 }, // alice
        glql_s2_s7: { count: 145, totalTimeToMerge: 98000000 }, // bob
        glql_s2_s8: { count: 220, totalTimeToMerge: 150000000 }, // charlie

        // Time segment 3 (2025-01-01 to 2025-04-01) with 3 users
        glql_s3_s6: { count: 165, totalTimeToMerge: 115000000 }, // alice
        glql_s3_s7: { count: 132, totalTimeToMerge: 89000000 }, // bob
        glql_s3_s8: { count: 198, totalTimeToMerge: 135000000 }, // charlie

        // Time segment 4 (2025-04-01 to 2025-07-01) with 3 users
        glql_s4_s6: { count: 142, totalTimeToMerge: 98000000 }, // alice
        glql_s4_s7: { count: 118, totalTimeToMerge: 82000000 }, // bob
        glql_s4_s8: { count: 175, totalTimeToMerge: 120000000 }, // charlie

        // Time segment 5 (2025-07-01 to 2025-08-08) with 3 users
        glql_s5_s6: { count: 89, totalTimeToMerge: 62000000 }, // alice
        glql_s5_s7: { count: 76, totalTimeToMerge: 54000000 }, // bob
        glql_s5_s8: { count: 112, totalTimeToMerge: 78000000 }, // charlie
      },
    };
  }

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  function mockDataIssues(): Record<string, any> {
    return {
      project: {
        __typename: 'Project',
        glql_s1: { __typename: 'IssueConnection', count: 0 },
        glql_s2: { __typename: 'IssueConnection', count: 0 },
        glql_s3: { __typename: 'IssueConnection', count: 0 },
        glql_s4: { __typename: 'IssueConnection', count: 0 },
        glql_s5: { __typename: 'IssueConnection', count: 0 },
        glql_s6: { __typename: 'IssueConnection', count: 0 },
        glql_s7: { __typename: 'IssueConnection', count: 2 },
        glql_s8: { __typename: 'IssueConnection', count: 35 },
        glql_s9: { __typename: 'IssueConnection', count: 0 },
        glql_s10: { __typename: 'IssueConnection', count: 0 },
        glql_s11: { __typename: 'IssueConnection', count: 0 },
        glql_s12: { __typename: 'IssueConnection', count: 5 },
        glql_s13: { __typename: 'IssueConnection', count: 2 },
      },
    };
  }

  test('transform aggregated data merge requests', () => {
    const query =
      "project = 'gitlab-org/gitlab' and type = MergeRequest and merged >= '2024-08-09' and merged <= '2025-08-08'";
    const dimensions = "timeSegment(1m) on mergedAt as 'Date merged'";
    const metrics =
      "count as 'Total count', totalTimeToMerge as 'Total time to merge', meanTimeToMerge as 'Mean time to merge'";
    compileWithAggregate(query, dimensions, metrics);

    const data = mockDataMergeRequests();
    const result = transformAggregatedData(data, dimensions, metrics);

    const dimension = {
      field: { key: 'mergedAt', label: 'Date merged', name: 'mergedAt' },
      fn: {
        quantity: 1,
        timeSegmentType: 'fromStartOfUnit',
        type: 'time',
        unit: 'm',
      },
    };

    const expectedData = {
      __typename: 'GlqlAggregatedDataConnection',
      count: 13,
      nodes: [
        {
          __typename: 'GlqlAggregatedData',
          count: 1876,
          meanTimeToMerge: 636685.6099850804,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's1',
            range: { from: '2024-08-09', to: '2024-09-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 1194422204.332011,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 2650,
          meanTimeToMerge: 664742.1877310819,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's2',
            range: { from: '2024-09-01', to: '2024-10-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 1761566797.487367,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 3103,
          meanTimeToMerge: 672438.3270561369,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's3',
            range: { from: '2024-10-01', to: '2024-11-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 2086576128.855193,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 2542,
          meanTimeToMerge: 653897.3717568328,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's4',
            range: { from: '2024-11-01', to: '2024-12-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 1662207119.005869,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 2037,
          meanTimeToMerge: 677879.7588408464,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's5',
            range: { from: '2024-12-01', to: '2025-01-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 1380841068.758804,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 2572,
          meanTimeToMerge: 875289.9823912473,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's6',
            range: { from: '2025-01-01', to: '2025-02-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 2251245834.710288,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 2716,
          meanTimeToMerge: 716577.0100979555,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's7',
            range: { from: '2025-02-01', to: '2025-03-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 1946223159.426047,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 2894,
          meanTimeToMerge: 698322.7407812944,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's8',
            range: { from: '2025-03-01', to: '2025-04-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 2020946011.821066,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 2993,
          meanTimeToMerge: 675346.6454370619,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's9',
            range: { from: '2025-04-01', to: '2025-05-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 2021312509.793126,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 2649,
          meanTimeToMerge: 692646.3960147984,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's10',
            range: { from: '2025-05-01', to: '2025-06-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 1834820303.043201,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 2762,
          meanTimeToMerge: 736609.8615398762,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's11',
            range: { from: '2025-06-01', to: '2025-07-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 2034516437.573138,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 3114,
          meanTimeToMerge: 618661.0495762178,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's12',
            range: { from: '2025-07-01', to: '2025-08-01' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 1926510508.380342,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 667,
          meanTimeToMerge: 676565.3345851994,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's13',
            range: { from: '2025-08-01', to: '2025-08-08' },
            dimension,
            type: 'time',
          },
          totalTimeToMerge: 451269078.168328,
        },
      ],
    };

    const expectedFields = [
      { key: 'mergedAt', label: 'Date merged', name: 'mergedAt' },
      { key: 'count', label: 'Total count', name: 'count' },
      { key: 'totalTimeToMerge', label: 'Total time to merge', name: 'totalTimeToMerge' },
      { key: 'meanTimeToMerge', label: 'Mean time to merge', name: 'meanTimeToMerge' },
    ];

    expect(jsonify(result.data)).toEqual(expectedData);
    expect(jsonify(result.fields)).toEqual(expectedFields);

    // ensure title is also present (computed)
    expect(
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      (result.data as any).nodes.map((node: any) => node.mergedAt.title),
    ).toEqual([
      'From 2024-08-09 to 2024-09-01',
      'From 2024-09-01 to 2024-10-01',
      'From 2024-10-01 to 2024-11-01',
      'From 2024-11-01 to 2024-12-01',
      'From 2024-12-01 to 2025-01-01',
      'From 2025-01-01 to 2025-02-01',
      'From 2025-02-01 to 2025-03-01',
      'From 2025-03-01 to 2025-04-01',
      'From 2025-04-01 to 2025-05-01',
      'From 2025-05-01 to 2025-06-01',
      'From 2025-06-01 to 2025-07-01',
      'From 2025-07-01 to 2025-08-01',
      'From 2025-08-01 to 2025-08-08',
    ]);
  });

  test('transform aggregated data issues', () => {
    const query =
      "project = 'gitlab-org/gitlab' and type = Issue and created >= '2024-08-08' and created <= '2025-08-08'";
    const dimensions = "timeSegment(1m) on created as 'Date created'";
    const metrics = 'count';
    compileWithAggregate(query, dimensions, metrics);

    const data = mockDataIssues();
    const result = transformAggregatedData(data, dimensions, metrics);

    const dimension = {
      field: { key: 'created', label: 'Date created', name: 'createdAt' },
      fn: {
        quantity: 1,
        timeSegmentType: 'fromStartOfUnit',
        type: 'time',
        unit: 'm',
      },
    };

    expect(jsonify(result.data)).toEqual({
      __typename: 'GlqlAggregatedDataConnection',
      count: 13,
      nodes: [
        {
          __typename: 'IssueConnection',
          count: 0,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's1',
            range: {
              from: '2024-08-08',
              to: '2024-09-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's1',
            range: {
              from: '2024-08-08',
              to: '2024-09-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 0,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's2',
            range: {
              from: '2024-09-01',
              to: '2024-10-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's2',
            range: {
              from: '2024-09-01',
              to: '2024-10-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 0,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's3',
            range: {
              from: '2024-10-01',
              to: '2024-11-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's3',
            range: {
              from: '2024-10-01',
              to: '2024-11-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 0,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's4',
            range: {
              from: '2024-11-01',
              to: '2024-12-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's4',
            range: {
              from: '2024-11-01',
              to: '2024-12-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 0,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's5',
            range: {
              from: '2024-12-01',
              to: '2025-01-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's5',
            range: {
              from: '2024-12-01',
              to: '2025-01-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 0,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's6',
            range: {
              from: '2025-01-01',
              to: '2025-02-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's6',
            range: {
              from: '2025-01-01',
              to: '2025-02-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 2,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's7',
            range: {
              from: '2025-02-01',
              to: '2025-03-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's7',
            range: {
              from: '2025-02-01',
              to: '2025-03-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 35,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's8',
            range: {
              from: '2025-03-01',
              to: '2025-04-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's8',
            range: {
              from: '2025-03-01',
              to: '2025-04-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 0,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's9',
            range: {
              from: '2025-04-01',
              to: '2025-05-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's9',
            range: {
              from: '2025-04-01',
              to: '2025-05-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 0,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's10',
            range: {
              from: '2025-05-01',
              to: '2025-06-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's10',
            range: {
              from: '2025-05-01',
              to: '2025-06-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 0,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's11',
            range: {
              from: '2025-06-01',
              to: '2025-07-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's11',
            range: {
              from: '2025-06-01',
              to: '2025-07-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 5,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's12',
            range: {
              from: '2025-07-01',
              to: '2025-08-01',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's12',
            range: {
              from: '2025-07-01',
              to: '2025-08-01',
            },
            dimension,
            type: 'time',
          },
        },
        {
          __typename: 'IssueConnection',
          count: 2,
          created: {
            __typename: 'GlqlDimensionSegment',
            key: 's13',
            range: {
              from: '2025-08-01',
              to: '2025-08-08',
            },
            dimension,
            type: 'time',
          },
          createdAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's13',
            range: {
              from: '2025-08-01',
              to: '2025-08-08',
            },
            dimension,
            type: 'time',
          },
        },
      ],
    });

    expect(jsonify(result.fields)).toEqual([
      {
        key: 'created',
        label: 'Date created',
        name: 'createdAt',
      },
      {
        key: 'count',
        label: 'Count',
        name: 'count',
      },
    ]);
  });

  test('transform aggregated data merge requests with multiple dimensions', () => {
    const query =
      "project = 'gitlab-org/gitlab' and type = MergeRequest and merged >= '2024-08-09' and merged <= '2025-08-08'";
    const dimensions =
      "timeSegment(3m) on mergedAt as 'Date merged', users(@alice, @bob, @charlie) on author as 'Author'";
    const metrics = "count as 'Total count', totalTimeToMerge as 'Total time to merge'";
    compileWithAggregate(query, dimensions, metrics);

    const data = mockDataMergeRequestsTwoDimensions();
    const result = transformAggregatedData(data, dimensions, metrics);

    const timeDimension = {
      field: { key: 'mergedAt', label: 'Date merged', name: 'mergedAt' },
      fn: {
        quantity: 3,
        timeSegmentType: 'fromStartOfUnit',
        type: 'time',
        unit: 'm',
      },
    };

    const userDimension = jsonify({
      field: { key: 'author', label: 'Author', name: 'author' },
      fn: {
        type: 'user',
        users: ['alice', 'bob', 'charlie'].map(
          (user) => new Value.Reference(ReferenceType.User, user),
        ),
      },
    });

    const user = (user: string) => jsonify(new Value.Reference(ReferenceType.User, user));

    expect(jsonify(result.data)).toEqual({
      __typename: 'GlqlAggregatedDataConnection',
      count: 15,
      nodes: [
        {
          __typename: 'GlqlAggregatedData',
          count: 125,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's1',
            range: { from: '2024-08-09', to: '2024-10-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's6',
            user: user('alice'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 85000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 98,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's1',
            range: { from: '2024-08-09', to: '2024-10-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's7',
            user: user('bob'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 72000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 156,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's1',
            range: { from: '2024-08-09', to: '2024-10-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's8',
            user: user('charlie'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 110000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 180,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's2',
            range: { from: '2024-10-01', to: '2025-01-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's6',
            user: user('alice'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 125000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 145,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's2',
            range: { from: '2024-10-01', to: '2025-01-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's7',
            user: user('bob'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 98000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 220,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's2',
            range: { from: '2024-10-01', to: '2025-01-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's8',
            user: user('charlie'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 150000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 165,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's3',
            range: { from: '2025-01-01', to: '2025-04-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's6',
            user: user('alice'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 115000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 132,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's3',
            range: { from: '2025-01-01', to: '2025-04-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's7',
            user: user('bob'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 89000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 198,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's3',
            range: { from: '2025-01-01', to: '2025-04-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's8',
            user: user('charlie'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 135000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 142,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's4',
            range: { from: '2025-04-01', to: '2025-07-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's6',
            user: user('alice'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 98000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 118,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's4',
            range: { from: '2025-04-01', to: '2025-07-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's7',
            user: user('bob'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 82000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 175,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's4',
            range: { from: '2025-04-01', to: '2025-07-01' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's8',
            user: user('charlie'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 120000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 89,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's5',
            range: { from: '2025-07-01', to: '2025-08-08' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's6',
            user: user('alice'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 62000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 76,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's5',
            range: { from: '2025-07-01', to: '2025-08-08' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's7',
            user: user('bob'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 54000000,
        },
        {
          __typename: 'GlqlAggregatedData',
          count: 112,
          mergedAt: {
            __typename: 'GlqlDimensionSegment',
            key: 's5',
            range: { from: '2025-07-01', to: '2025-08-08' },
            dimension: timeDimension,
            type: 'time',
          },
          author: {
            __typename: 'GlqlDimensionSegment',
            key: 's8',
            user: user('charlie'),
            dimension: userDimension,
            type: 'user',
          },
          totalTimeToMerge: 78000000,
        },
      ],
    });

    expect(jsonify(result.fields)).toEqual([
      { key: 'mergedAt', label: 'Date merged', name: 'mergedAt' },
      { key: 'author', label: 'Author', name: 'author' },
      { key: 'count', label: 'Total count', name: 'count' },
      {
        key: 'totalTimeToMerge',
        label: 'Total time to merge',
        name: 'totalTimeToMerge',
      },
    ]);

    // ensure titles are also present (computed)
    expect(
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      (result.data as any).nodes.map((node: any) => [node.mergedAt.title, node.author.title]),
    ).toEqual([
      ['From 2024-08-09 to 2024-10-01', '@alice'],
      ['From 2024-08-09 to 2024-10-01', '@bob'],
      ['From 2024-08-09 to 2024-10-01', '@charlie'],

      ['From 2024-10-01 to 2025-01-01', '@alice'],
      ['From 2024-10-01 to 2025-01-01', '@bob'],
      ['From 2024-10-01 to 2025-01-01', '@charlie'],

      ['From 2025-01-01 to 2025-04-01', '@alice'],
      ['From 2025-01-01 to 2025-04-01', '@bob'],
      ['From 2025-01-01 to 2025-04-01', '@charlie'],

      ['From 2025-04-01 to 2025-07-01', '@alice'],
      ['From 2025-04-01 to 2025-07-01', '@bob'],
      ['From 2025-04-01 to 2025-07-01', '@charlie'],

      ['From 2025-07-01 to 2025-08-08', '@alice'],
      ['From 2025-07-01 to 2025-08-08', '@bob'],
      ['From 2025-07-01 to 2025-08-08', '@charlie'],
    ]);
  });
});
