import { createHash } from 'node:crypto';
import * as fs from 'fs';
import path from 'node:path';
import { compile as originalCompile, transform } from '../../src/lib';
import { resetUniqueIdCounters } from '../../src/utils/common';
import { featureFlags } from '../../src/utils/feature_flags';
import { CompileContext } from '../../src/types/compile_context';
import { CompileOutput } from '../../src/types/compile_output';

function dumpGraphQL(output: string) {
  if (process.env['DUMP_GRAPHQL']) {
    const sha = createHash('sha256').update(output).digest('hex');
    const dir = path.join(process.cwd(), 'tmp/ts/graphql');
    fs.mkdirSync(dir, { recursive: true });
    fs.writeFileSync(path.join(dir, `${sha.substring(0, 8)}.graphql`), output);
  }
}

jest.mock('../../src/lib', () => {
  const original = jest.requireActual('../../src/lib');
  return {
    ...original,
    compile: (query: string, context: CompileContext): CompileOutput => {
      const output = original.compile(query, context);
      if (output.success) dumpGraphQL(output.output);
      return output;
    },
  };
});

function compile(query: string, context: CompileContext): CompileOutput {
  const output = originalCompile(query, context);
  if (output.success) dumpGraphQL(output.output);
  return output;
}

beforeEach(() => {
  featureFlags.glqlWorkItems = false;
  resetUniqueIdCounters();
});

afterEach(() => {
  featureFlags.glqlWorkItems = false;
  resetUniqueIdCounters();
});

export function compileGraphQL(query: string): string {
  return compile(query, { fields: 'id, title' }).output;
}

export function compileWithUserContext(username: string, query: string): string {
  return compile(query, { fields: 'id, title', username }).output;
}

export function compileWithGroupContext(group: string, query: string): string {
  return compile(query, { fields: 'id, title', group }).output;
}

export function compileWithProjectContext(project: string, query: string): string {
  return compile(query, { fields: 'id, title', project }).output;
}

export function compileWithTimeFromString(referenceTime: string, query: string): string {
  jest.useFakeTimers().setSystemTime(new Date(referenceTime));

  return compile(query, { fields: 'id, title' }).output;
}

export function compileWithSort(query: string, sort: string): string {
  return compile(query, { sort, fields: 'id, title' }).output;
}

export function compileWithFields(query: string, fields: string): string {
  return compile(query, { fields }).output;
}

export function compileWithAggregate(
  query: string,
  groupBy: string,
  aggregate: string,
): CompileOutput {
  return compile(query, { groupBy, aggregate });
}

export function normalizeGraphQL(graphql: string): string {
  return graphql.replace(/\n/g, ' ').replace(/\s+/g, ' ').trim();
}

export function randomizeCase(s: string): string {
  return s
    .split('')
    .map((c) => (Math.random() < 0.5 ? c.toUpperCase() : c.toLowerCase()))
    .join('');
}

export function transformData(data: unknown, fields: string) {
  return transform(data, { fields });
}

export function transformAggregatedData(data: unknown, groupBy: string, aggregate: string) {
  return transform(data, { groupBy, aggregate, fields: 'id, title' });
}

export function jsonify(input: unknown): unknown {
  return JSON.parse(JSON.stringify(input));
}
