import { featureFlags } from '../../src/utils/feature_flags';
import { compileWithSort } from './test_helpers';

function testSort(sort: string, query: string, expected: string[], expectedDesc: string[]) {
  // test default
  expect(compileWithSort(`group = "gitlab-org" and ${query}`, sort).split('\n')).toEqual(
    expect.arrayContaining(expected),
  );

  // test ascending
  expect(compileWithSort(`group = "gitlab-org" and ${query}`, `${sort} asc`).split('\n')).toEqual(
    expect.arrayContaining(expected),
  );

  // test descending
  expect(compileWithSort(`group = "gitlab-org" and ${query}`, `${sort} desc`).split('\n')).toEqual(
    expect.arrayContaining(expectedDesc),
  );
}

test.each([
  [
    'merge requests sort by merged',
    'merged',
    'type = MergeRequest and author = @foo',
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: MERGED_AT_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: MERGED_AT_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'merge requests sort by closed',
    'closed',
    'type = MergeRequest and author = @foo',
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CLOSED_AT_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CLOSED_AT_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'merge requests sort by title',
    'title',
    'type = MergeRequest and author = @foo',
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: TITLE_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: TITLE_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'merge requests sort by popularity',
    'popularity',
    'type = MergeRequest and author = @foo',
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: POPULARITY_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: POPULARITY_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'merge requests sort by milestone',
    'milestone',
    'type = MergeRequest and author = @foo',
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: MILESTONE_DUE_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: MILESTONE_DUE_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'merge requests sort by updated',
    'updated',
    'type = MergeRequest and author = @foo',
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: UPDATED_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: UPDATED_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'merge requests sort by created',
    'created',
    'type = MergeRequest and author = @foo',
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CREATED_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CREATED_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'epics sort by start',
    'start',
    'type = Epic and author = @foo',
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: START_DATE_ASC',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: START_DATE_DESC',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
  ],
  [
    'epics sort by due',
    'due',
    'type = Epic and author = @foo',
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: DUE_DATE_ASC',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: DUE_DATE_DESC',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
  ],
  [
    'epics sort by title',
    'title',
    'type = Epic and author = @foo',
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: TITLE_ASC',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: TITLE_DESC',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
  ],
  [
    'epics sort by created',
    'created',
    'type = Epic and author = @foo',
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CREATED_ASC',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CREATED_DESC',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
  ],
  [
    'epics sort by updated',
    'updated',
    'type = Epic and author = @foo',
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: UPDATED_ASC',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: UPDATED_DESC',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by due',
    'due',
    'type = Issue and author = @foo',
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: DUE_DATE_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: DUE_DATE_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by title',
    'title',
    'type = Issue and author = @foo',
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: TITLE_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: TITLE_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by popularity',
    'popularity',
    'type = Issue and author = @foo',
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: POPULARITY_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: POPULARITY_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by closed',
    'closed',
    'type = Issue and author = @foo',
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CLOSED_AT_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CLOSED_AT_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by weight',
    'weight',
    'type = Issue and author = @foo',
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: WEIGHT_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: WEIGHT_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by health',
    'health',
    'type = Issue and author = @foo',
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: HEALTH_STATUS_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: HEALTH_STATUS_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by milestone',
    'milestone',
    'type = Issue and author = @foo',
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: MILESTONE_DUE_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: MILESTONE_DUE_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by updated',
    'updated',
    'type = Issue and author = @foo',
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: UPDATED_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: UPDATED_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by created',
    'created',
    'type = Issue and author = @foo',
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CREATED_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CREATED_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by due date alias',
    'dueDate',
    'type = Issue and author = @foo',
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: DUE_DATE_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: DUE_DATE_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by created at alias',
    'createdAt',
    'type = Issue and author = @foo',
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CREATED_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    issues(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CREATED_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
  [
    'merge requests sort by merged at alias',
    'mergedAt',
    'type = MergeRequest and author = @foo',
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: MERGED_AT_ASC',
      '      includeSubgroups: true',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: MERGED_AT_DESC',
      '      includeSubgroups: true',
      '    ) {',
    ],
  ],
])('%s', (_, sort, query, expected, expectedDesc) => {
  testSort(sort, query, expected, expectedDesc);
});

test.each([
  [
    'issues sort by due date (work items)',
    'start',
    'type = Issue AND author = @foo',
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: START_DATE_ASC',
      '      includeDescendants: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: START_DATE_DESC',
      '      includeDescendants: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by title (work items)',
    'title',
    'type = Issue AND author = @foo',
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: TITLE_ASC',
      '      includeDescendants: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: TITLE_DESC',
      '      includeDescendants: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by popularity (work items)',
    'popularity',
    'type = Issue AND author = @foo',
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: POPULARITY_ASC',
      '      includeDescendants: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: POPULARITY_DESC',
      '      includeDescendants: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by closed (work items)',
    'closed',
    'type = Issue AND author = @foo',
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CLOSED_AT_ASC',
      '      includeDescendants: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CLOSED_AT_DESC',
      '      includeDescendants: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by weight (work items)',
    'weight',
    'type = Issue AND author = @foo',
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: WEIGHT_ASC',
      '      includeDescendants: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: WEIGHT_DESC',
      '      includeDescendants: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by health (work items)',
    'health',
    'type = Issue AND author = @foo',
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: HEALTH_STATUS_ASC',
      '      includeDescendants: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: HEALTH_STATUS_DESC',
      '      includeDescendants: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by milestone (work items)',
    'milestone',
    'type = Issue AND author = @foo',
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: MILESTONE_DUE_ASC',
      '      includeDescendants: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: MILESTONE_DUE_DESC',
      '      includeDescendants: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by updated (work items)',
    'updated',
    'type = Issue AND author = @foo',
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: UPDATED_ASC',
      '      includeDescendants: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: UPDATED_DESC',
      '      includeDescendants: true',
      '    ) {',
    ],
  ],
  [
    'issues sort by created (work items)',
    'created',
    'type = Issue AND author = @foo',
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CREATED_ASC',
      '      includeDescendants: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: ISSUE',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      sort: CREATED_DESC',
      '      includeDescendants: true',
      '    ) {',
    ],
  ],
])('%s', (_, sort, query, expected, expectedDesc) => {
  featureFlags.glqlWorkItems = true;

  testSort(sort, query, expected, expectedDesc);
});

test.each`
  type              | errorMessage
  ${'MergeRequest'} | ${'Error: `invalid` is not a recognized sort field for merge requests. Supported fields to sort by: `merged`, `closed`, `title`, `popularity`, `milestone`, `updated`, `created`.'}
  ${'Epic'}         | ${'Error: `invalid` is not a recognized sort field for work items. Supported fields to sort by: `due`, `title`, `popularity`, `closed`, `weight`, `health`, `milestone`, `updated`, `created`, `start`.'}
  ${'Issue'}        | ${'Error: `invalid` is not a recognized sort field for work items. Supported fields to sort by: `due`, `title`, `popularity`, `closed`, `weight`, `health`, `milestone`, `updated`, `created`.'}
`('sort $type by invalid field', ({ type, errorMessage }) => {
  expect(compileWithSort(`group = "gitlab-org" and type = ${type}`, 'invalid')).toEqual(
    errorMessage,
  );
});

test('sort by invalid order', () => {
  expect(compileWithSort("group = 'gitlab-org' and type = MergeRequest", 'title invalid')).toEqual(
    'Error: Invalid sort order: `invalid`. Valid sort orders: `asc`, `desc`.',
  );
});
