import { compile } from '../../src/lib';
import { featureFlags } from '../../src/utils/feature_flags';
import { compileGraphQL, jsonify } from './test_helpers';
import { CompileContext } from '../../src/types/compile_context';

test.each([
  [
    'subscribed = true',
    [
      '    issues(',
      '      subscribed: EXPLICITLY_SUBSCRIBED',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      subscribed: EXPLICITLY_SUBSCRIBED',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      subscribed: EXPLICITLY_SUBSCRIBED',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      subscribed: EXPLICITLY_SUBSCRIBED',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'subscribed = false',
    [
      '    issues(',
      '      subscribed: EXPLICITLY_UNSUBSCRIBED',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      subscribed: EXPLICITLY_UNSUBSCRIBED',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      subscribed: EXPLICITLY_UNSUBSCRIBED',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      subscribed: EXPLICITLY_UNSUBSCRIBED',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'myreaction = "thumbsup"',
    [
      '    issues(',
      '      myReactionEmoji: "thumbsup"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      myReactionEmoji: "thumbsup"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      myReactionEmoji: "thumbsup"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      myReactionEmoji: "thumbsup"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'myreaction != "thumbsup"',
    [
      '    issues(',
      '      not: {myReactionEmoji: "thumbsup"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      not: {myReactionEmoji: "thumbsup"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      not: {myReactionEmoji: "thumbsup"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    mergeRequests(',
      '      not: {myReactionEmoji: "thumbsup"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
])('%s', (query, expectedIssues, expectedEpics, expectedWorkItems, expectedMergeRequests) => {
  expect(compileGraphQL(`project = 'gitlab-org/gitlab' and ${query}`).split('\n')).toEqual(
    expect.arrayContaining(expectedIssues),
  );

  expect(compileGraphQL(`group = 'gitlab-org' and type = Epic and ${query}`).split('\n')).toEqual(
    expect.arrayContaining(expectedEpics),
  );

  featureFlags.glqlWorkItems = true;

  expect(compileGraphQL(`project = 'gitlab-org/gitlab' and ${query}`).split('\n')).toEqual(
    expect.arrayContaining(expectedWorkItems),
  );

  featureFlags.glqlWorkItems = false;

  expect(
    compileGraphQL(`project = 'gitlab-org/gitlab' and type = MergeRequest and ${query}`).split(
      '\n',
    ),
  ).toEqual(expect.arrayContaining(expectedMergeRequests));
});

test.each([
  [
    'subscribed = "invalid"',
    'Error: `subscribed` cannot be compared with `"invalid"`. Supported value types: `Boolean` (`true`, `false`).',
  ],
  [
    'myreaction = ~"invalid"',
    'Error: `myreaction` cannot be compared with `~invalid`. Supported value types: `String`.',
  ],
])('%s', (query, expectedError) => {
  expect(compileGraphQL(`project = 'gitlab-org/gitlab' and ${query}`)).toEqual(expectedError);
});

test('fields in output', () => {
  const query = "project = 'gitlab-org/gitlab' and type = Issue";
  const context: CompileContext = {
    fields: "title, updated, author, assignees, labels, labels('After*'), lastComment",
  };

  const output = compile(query, context);

  expect(jsonify(output.fields)).toEqual([
    { key: 'title', label: 'Title', name: 'title' },
    { key: 'updated', label: 'Updated', name: 'updatedAt' },
    { key: 'author', label: 'Author', name: 'author' },
    { key: 'assignees', label: 'Assignees', name: 'assignees' },
    { key: 'labels', label: 'Labels', name: 'labels' },
    { key: 'labels_After*', label: 'Labels: After', name: 'labels' },
    { key: 'lastComment', label: 'Last comment', name: 'lastComment' },
  ]);
});

test('fields in output with aggregate', () => {
  const query =
    'project = "gitlab-org/gitlab" and type = MergeRequest and merged >= 2025-01-01 and merged <= 2025-01-02';
  const context: CompileContext = {
    fields: 'id, title',
    groupBy: "timeSegment(2d) on merged as 'Date merged'",
    aggregate: "count as 'Total MRs', totalTimeToMerge as 'TTTM'",
  };

  const output = compile(query, context);

  expect(jsonify(output.fields)).toEqual([
    { key: 'merged', label: 'Date merged', name: 'mergedAt' },
    { key: 'count', label: 'Total MRs', name: 'count' },
    { key: 'totalTimeToMerge', label: 'TTTM', name: 'totalTimeToMerge' },
  ]);
});
