import { compileGraphQL } from './test_helpers';

test.each([
  ['id = 1', ['    mergeRequests(iids: "1", before: $before, after: $after, first: $limit) {']],
  [
    'id in (1, 2, 3)',
    [
      '    mergeRequests(',
      '      iids: ["1", "2", "3"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'assignee = @johndoe',
    [
      '    mergeRequests(',
      '      assigneeUsername: "johndoe"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'assignee = "foo"',
    [
      '    mergeRequests(',
      '      assigneeUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'assignee = any',
    [
      '    mergeRequests(',
      '      assigneeWildcardId: ANY',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'assignee != any',
    [
      '    mergeRequests(',
      '      assigneeWildcardId: NONE',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'assignee = none',
    [
      '    mergeRequests(',
      '      assigneeWildcardId: NONE',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'assignee != none',
    [
      '    mergeRequests(',
      '      assigneeWildcardId: ANY',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'assignee != @Johndoe',
    [
      '    mergeRequests(',
      '      not: {assigneeUsernames: "Johndoe"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'reviewer = @johndoe',
    [
      '    mergeRequests(',
      '      reviewerUsername: "johndoe"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'reviewer = "foo"',
    [
      '    mergeRequests(',
      '      reviewerUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'reviewer = any',
    [
      '    mergeRequests(',
      '      reviewerWildcardId: ANY',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'reviewer != any',
    [
      '    mergeRequests(',
      '      reviewerWildcardId: NONE',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'reviewer = none',
    [
      '    mergeRequests(',
      '      reviewerWildcardId: NONE',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'reviewer != none',
    [
      '    mergeRequests(',
      '      reviewerWildcardId: ANY',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'reviewer != @Johndoe',
    [
      '    mergeRequests(',
      '      not: {reviewerUsername: "Johndoe"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'author = @johndoe',
    [
      '    mergeRequests(',
      '      authorUsername: "johndoe"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'author != "foo"',
    [
      '    mergeRequests(',
      '      not: {authorUsername: "foo"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'approver = @johndoe',
    [
      '    mergeRequests(',
      '      approvedBy: "johndoe"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'approver != "foo"',
    [
      '    mergeRequests(',
      '      not: {approvedBy: "foo"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'approver != (@foo, @bar)',
    [
      '    mergeRequests(',
      '      not: {approvedBy: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'merger = @johndoe',
    [
      '    mergeRequests(',
      '      mergedBy: "johndoe"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'merger = "foo"',
    ['    mergeRequests(mergedBy: "foo", before: $before, after: $after, first: $limit) {'],
  ],
  [
    'draft = true',
    ['    mergeRequests(draft: true, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'draft = false',
    ['    mergeRequests(draft: false, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'state = OPENED',
    ['    mergeRequests(state: opened, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'state = closed',
    ['    mergeRequests(state: closed, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'state = merged',
    ['    mergeRequests(state: merged, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'state = ALL',
    ['    mergeRequests(state: all, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'label = "foo"',
    ['    mergeRequests(labelName: "foo", before: $before, after: $after, first: $limit) {'],
  ],
  [
    'label = ~foo',
    ['    mergeRequests(labelName: "foo", before: $before, after: $after, first: $limit) {'],
  ],
  [
    'label = ("foo", "bar")',
    [
      '    mergeRequests(',
      '      labelName: ["foo", "bar"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label != (~foo, ~bar)',
    [
      '    mergeRequests(',
      '      not: {labelName: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'environment = "gprd"',
    [
      '    mergeRequests(',
      '      environmentName: "gprd"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'milestone = %12.4',
    [
      '    mergeRequests(',
      '      milestoneTitle: "12.4"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'milestone = any',
    [
      '    mergeRequests(',
      '      milestoneWildcardId: ANY',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'milestone = none',
    [
      '    mergeRequests(',
      '      milestoneWildcardId: NONE',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'milestone != none',
    [
      '    mergeRequests(',
      '      milestoneWildcardId: ANY',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
])('%s', (query, expected) => {
  expect(
    compileGraphQL(`project = "gitlab-org/gitlab" and type = MergeRequest and ${query}`).split(
      '\n',
    ),
  ).toEqual(expect.arrayContaining(expected));
});

describe.each`
  field
  ${'sourceBranch'}
  ${'targetBranch'}
`('$field = ', ({ field }) => {
  test.each([
    [
      `${field} = "foo"`,
      [
        '    mergeRequests(',
        `      ${field}es: "foo"`,
        '      before: $before',
        '      after: $after',
        '      first: $limit',
        '    ) {',
      ],
    ],
    [
      `${field} != "foo"`,
      [
        '    mergeRequests(',
        `      not: {${field}es: "foo"}`,
        '      before: $before',
        '      after: $after',
        '      first: $limit',
        '    ) {',
      ],
    ],
    [
      `${field} in ("foo", "bar")`,
      [
        '    mergeRequests(',
        `      ${field}es: ["foo", "bar"]`,
        '      before: $before',
        '      after: $after',
        '      first: $limit',
        '    ) {',
      ],
    ],
    [
      `${field} != ("foo", "bar")`,
      [
        '    mergeRequests(',
        `      not: {${field}es: ["foo", "bar"]}`,
        '      before: $before',
        '      after: $after',
        '      first: $limit',
        '    ) {',
      ],
    ],
  ])('%s', (query, expected) => {
    expect(
      compileGraphQL(`project = "gitlab-org/gitlab" and type = MergeRequest and ${query}`).split(
        '\n',
      ),
    ).toEqual(expect.arrayContaining(expected));
  });
});

describe.each`
  field
  ${'created'}
  ${'merged'}
  ${'deployed'}
  ${'updated'}
`('$field', ({ field }) => {
  test.each([
    [
      `${field} > 2020-01-01`,
      [
        '    mergeRequests(',
        `      ${field}After: "2020-01-01 23:59"`,
        '      before: $before',
        '      after: $after',
        '      first: $limit',
        '    ) {',
      ],
    ],
    [
      `${field} < 2020-01-01`,
      [
        '    mergeRequests(',
        `      ${field}Before: "2020-01-01 00:00"`,
        '      before: $before',
        '      after: $after',
        '      first: $limit',
        '    ) {',
      ],
    ],
    [
      `${field} >= 2020-01-01`,
      [
        '    mergeRequests(',
        `      ${field}After: "2020-01-01 00:00"`,
        '      before: $before',
        '      after: $after',
        '      first: $limit',
        '    ) {',
      ],
    ],
    [
      `${field} <= 2020-01-01`,
      [
        '    mergeRequests(',
        `      ${field}Before: "2020-01-01 23:59"`,
        '      before: $before',
        '      after: $after',
        '      first: $limit',
        '    ) {',
      ],
    ],
  ])('%s', (query, expected) => {
    expect(
      compileGraphQL(`project = "gitlab-org/gitlab" and type = MergeRequest and ${query}`).split(
        '\n',
      ),
    ).toEqual(expect.arrayContaining(expected));
  });
});

test.each`
  query                              | errorMessage
  ${'id = (1, 2, 3)'}                | ${'Error: `id` does not support the equals (`=`) operator for `(1, 2, 3)`. Supported operators: is one of (`in`).'}
  ${'assignee = 123'}                | ${'Error: `assignee` cannot be compared with `123`. Supported value types: `String`, `Username` (example: `@username`), `Nullable` (`null`, `none`, `any`).'}
  ${'assignee = (@foo, @bar)'}       | ${'Error: `assignee` cannot be compared with `(@foo, @bar)`. Supported value types: `String`, `Username` (example: `@username`), `Nullable` (`null`, `none`, `any`).'}
  ${'assignee in (@foo, @bar)'}      | ${'Error: `assignee` does not support the is one of (`in`) operator. Supported operators: equals (`=`), not equals (`!=`).'}
  ${'reviewer = 123'}                | ${'Error: `reviewer` cannot be compared with `123`. Supported value types: `String`, `Username` (example: `@username`), `Nullable` (`null`, `none`, `any`).'}
  ${'reviewer = (@foo, @bar)'}       | ${'Error: `reviewer` cannot be compared with `(@foo, @bar)`. Supported value types: `String`, `Username` (example: `@username`), `Nullable` (`null`, `none`, `any`).'}
  ${'reviewer in (@foo, @bar)'}      | ${'Error: `reviewer` does not support the is one of (`in`) operator. Supported operators: equals (`=`), not equals (`!=`).'}
  ${'author = any'}                  | ${'Error: `author` cannot be compared with `any`. Supported value types: `String`, `Username` (example: `@username`).'}
  ${'approver in (@foo, @bar)'}      | ${'Error: `approver` does not support the is one of (`in`) operator. Supported operators: equals (`=`), not equals (`!=`).'}
  ${'merger in (@foo, @bar)'}        | ${'Error: `merger` does not support the is one of (`in`) operator. Supported operators: equals (`=`).'}
  ${'draft = "foo"'}                 | ${'Error: `draft` cannot be compared with `"foo"`. Supported value types: `Boolean` (`true`, `false`).'}
  ${'state = any'}                   | ${'Error: `state` cannot be compared with `any`. Supported value types: `Enum` (`opened`, `closed`, `merged`, `all`).'}
  ${'state != merged'}               | ${'Error: `state` does not support the not equals (`!=`) operator. Supported operators: equals (`=`).'}
  ${'created = true'}                | ${'Error: `created` cannot be compared with `true`. Supported value types: `Date`. Did you mean `state = opened`?'}
  ${'opened = true'}                 | ${'Error: `opened` cannot be compared with `true`. Supported value types: `Date`. Did you mean `state = opened`?'}
  ${'deployed = "foo"'}              | ${'Error: `deployed` cannot be compared with `"foo"`. Supported value types: `Date`.'}
  ${'environment = 123'}             | ${'Error: `environment` cannot be compared with `123`. Supported value types: `String`.'}
  ${"environment != 'foo'"}          | ${'Error: `environment` does not support the not equals (`!=`) operator. Supported operators: equals (`=`).'}
  ${'label in (~foo, ~bar)'}         | ${'Error: `label` does not support the is one of (`in`) operator. Supported operators: equals (`=`), not equals (`!=`).'}
  ${'milestone in (%12.4, %12.5)'}   | ${'Error: `milestone` does not support the is one of (`in`) operator. Supported operators: equals (`=`), not equals (`!=`).'}
  ${'sourceBranch = ("foo", "bar")'} | ${'Error: `sourceBranch` does not support the equals (`=`) operator for `("foo", "bar")`. Supported operators: is one of (`in`), not equals (`!=`).'}
  ${'targetBranch = ("foo", "bar")'} | ${'Error: `targetBranch` does not support the equals (`=`) operator for `("foo", "bar")`. Supported operators: is one of (`in`), not equals (`!=`).'}
`('$query', ({ query, errorMessage }) => {
  expect(
    compileGraphQL(`project = "gitlab-org/gitlab" and type = MergeRequest and ${query}`),
  ).toEqual(errorMessage);
});
