import { featureFlags } from '../../src/utils/feature_flags';
import { compileGraphQL } from './test_helpers';

test.each([
  [
    'label = "foo"',
    ['    issues(labelName: "foo", before: $before, after: $after, first: $limit) {'],
    ['    workItems(labelName: "foo", before: $before, after: $after, first: $limit) {'],
  ],
  [
    'label = ("foo", "bar")',
    ['    issues(labelName: ["foo", "bar"], before: $before, after: $after, first: $limit) {'],
    [
      '    workItems(',
      '      labelName: ["foo", "bar"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = ()',
    ['    issues(labelName: "NONE", before: $before, after: $after, first: $limit) {'],
    ['    workItems(labelName: "NONE", before: $before, after: $after, first: $limit) {'],
  ],
  [
    'label != ()',
    ['    issues(labelName: "ANY", before: $before, after: $after, first: $limit) {'],
    ['    workItems(labelName: "ANY", before: $before, after: $after, first: $limit) {'],
  ],
  [
    `label != "foo" and label != "bar" and label = "baz" and label = "quo" and label = ("bug", "enhancement") and label != ("feature", "maintenance")`,
    [
      '    issues(',
      '      labelName: ["baz", "quo", "bug", "enhancement"]',
      '      not: {labelName: ["foo", "bar", "feature", "maintenance"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      labelName: ["baz", "quo", "bug", "enhancement"]',
      '      not: {labelName: ["foo", "bar", "feature", "maintenance"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = null',
    ['    issues(labelName: "NONE", before: $before, after: $after, first: $limit) {'],
    ['    workItems(labelName: "NONE", before: $before, after: $after, first: $limit) {'],
  ],
  [
    'label in ("foo", "bar")',
    [
      '    issues(',
      '      or: {labelNames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      or: {labelNames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label in ("foo", "bar") and label = "baz"',
    [
      '    issues(',
      '      labelName: "baz"',
      '      or: {labelNames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      labelName: "baz"',
      '      or: {labelNames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = "foo" and label = "bar"',
    ['    issues(labelName: ["foo", "bar"], before: $before, after: $after, first: $limit) {'],
    [
      '    workItems(',
      '      labelName: ["foo", "bar"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = "foo" and label = "bar" and label = "baz"',
    [
      '    issues(',
      '      labelName: ["foo", "bar", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      labelName: ["foo", "bar", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = "foo" and label = ("bar", "boo")',
    [
      '    issues(',
      '      labelName: ["foo", "bar", "boo"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      labelName: ["foo", "bar", "boo"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = "foo" and label = ("bar", "boo") and label = "baz"',
    [
      '    issues(',
      '      labelName: ["foo", "bar", "boo", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      labelName: ["foo", "bar", "boo", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = ("foo", "bar") and label=("boo", "baz")',
    [
      '    issues(',
      '      labelName: ["foo", "bar", "boo", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      labelName: ["foo", "bar", "boo", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label != "foo"',
    ['    issues(not: {labelName: "foo"}, before: $before, after: $after, first: $limit) {'],
    [
      '    workItems(',
      '      not: {labelName: "foo"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = ~"foo"',
    ['    issues(labelName: "foo", before: $before, after: $after, first: $limit) {'],
    ['    workItems(labelName: "foo", before: $before, after: $after, first: $limit) {'],
  ],
  [
    'label = (~frontend, ~AI) and label=(~"1st contribution", ~"workflow::in review")',
    [
      '    issues(',
      '      labelName: ["frontend", "AI", "1st contribution", "workflow::in review"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      labelName: ["frontend", "AI", "1st contribution", "workflow::in review"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = ~foo and label = (~bar, ~baz)',
    [
      '    issues(',
      '      labelName: ["foo", "bar", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      labelName: ["foo", "bar", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = ~foo and label = ~bar',
    ['    issues(labelName: ["foo", "bar"], before: $before, after: $after, first: $limit) {'],
    [
      '    workItems(',
      '      labelName: ["foo", "bar"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = "foo" and label = (~bar, ~baz)',
    [
      '    issues(',
      '      labelName: ["foo", "bar", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      labelName: ["foo", "bar", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'label = "foo" and label = ("bar", ~baz) and label = ~quox',
    [
      '    issues(',
      '      labelName: ["foo", "bar", "baz", "quox"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      labelName: ["foo", "bar", "baz", "quox"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
])('%s', (query, expectedIssues, expectedWorkItems) => {
  expect(compileGraphQL(`project = "gitlab-org/gitlab" and ${query}`).split('\n')).toEqual(
    expect.arrayContaining(expectedIssues),
  );

  featureFlags.glqlWorkItems = true;

  expect(compileGraphQL(`project = "gitlab-org/gitlab" and ${query}`).split('\n')).toEqual(
    expect.arrayContaining(expectedWorkItems),
  );
});

const gt100Labels = new Array(101)
  .fill('label')
  .map((_, i) => `"label${i}"`)
  .join(', ');

test.each`
  query                              | expectedError
  ${'label = @username'}             | ${'Error: `label` cannot be compared with `@username`. Supported value types: `String`, `List`, `Nullable` (`null`, `none`, `any`), `Label` (example: `~frontend`).'}
  ${'label = ~"unterminated string'} | ${'Error: Invalid label reference `~"unterminated string`'}
  ${`label in (${gt100Labels})`}     | ${'Error: `label` field exceeds maximum limit of 100 items.'}
`('$query', ({ query, expectedError }) => {
  expect(compileGraphQL(`project = "gitlab-org/gitlab" and ${query}`)).toBe(expectedError);
});
