import { featureFlags } from '../../src/utils/feature_flags';
import { compileGraphQL } from './test_helpers';

test.each([
  [
    'iteration = 1234',
    ['    issues(iterationId: 1234, before: $before, after: $after, first: $limit) {'],
    ['    workItems(iterationId: 1234, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'iteration in (12, 34)',
    ['    issues(iterationId: [12, 34], before: $before, after: $after, first: $limit) {'],
    ['    workItems(iterationId: [12, 34], before: $before, after: $after, first: $limit) {'],
  ],
  [
    'iteration != (12, 34)',
    [
      '    issues(',
      '      not: {iterationId: [12, 34]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      not: {iterationId: [12, 34]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'cadence = 1234',
    [
      '    issues(',
      '      iterationCadenceId: "gid://gitlab/Iterations::Cadence/1234"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      iterationCadenceId: "gid://gitlab/Iterations::Cadence/1234"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'cadence in (12, 34)',
    [
      '    issues(',
      '      iterationCadenceId: ["gid://gitlab/Iterations::Cadence/12", "gid://gitlab/Iterations::Cadence/34"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      iterationCadenceId: ["gid://gitlab/Iterations::Cadence/12", "gid://gitlab/Iterations::Cadence/34"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'cadence = 1234 and iteration = current',
    [
      '    issues(',
      '      iterationCadenceId: "gid://gitlab/Iterations::Cadence/1234"',
      '      iterationWildcardId: CURRENT',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      iterationCadenceId: "gid://gitlab/Iterations::Cadence/1234"',
      '      iterationWildcardId: CURRENT',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'iteration = *iteration:1234',
    ['    issues(iterationId: "1234", before: $before, after: $after, first: $limit) {'],
    ['    workItems(iterationId: "1234", before: $before, after: $after, first: $limit) {'],
  ],
])('%s', (query, expectedIssues, expectedWorkItems) => {
  featureFlags.glqlWorkItems = false;

  expect(compileGraphQL(`project = "gitlab-org/gitlab" and ${query}`).split('\n')).toEqual(
    expect.arrayContaining(expectedIssues),
  );

  featureFlags.glqlWorkItems = true;

  expect(compileGraphQL(`project = "gitlab-org/gitlab" and ${query}`).split('\n')).toEqual(
    expect.arrayContaining(expectedWorkItems),
  );
});

test.each`
  query                                 | expectedError
  ${'iteration = (12, 34)'}             | ${'Error: `iteration` does not support the equals (`=`) operator for `(12, 34)`. Supported operators: is one of (`in`), not equals (`!=`).'}
  ${'cadence = (12, 34)'}               | ${'Error: `cadence` does not support the equals (`=`) operator for `(12, 34)`. Supported operators: is one of (`in`).'}
  ${'cadence != (12, 34)'}              | ${'Error: `cadence` does not support the not equals (`!=`) operator for `(12, 34)`. Supported operators: is one of (`in`).'}
  ${'iteration = ~frontend'}            | ${'Error: `iteration` cannot be compared with `~frontend`. Supported value types: `String`, `List`, `Number`, `Nullable` (`null`, `none`, `any`), `Enum` (`current`), `Iteration` (example: `*iteration:25263`.'}
  ${'iteration = *iteration:123foobar'} | ${'Error: Invalid iteration reference `*iteration:123foobar`'}
`('%s', ({ query, expectedError }) => {
  expect(compileGraphQL(`project = "gitlab-org/gitlab" and ${query}`)).toBe(expectedError);
});
