import { compileGraphQL, compileWithTimeFromString, compileWithUserContext } from './test_helpers';

test.each([
  [
    'current user with username',
    'assignee = currentUser()',
    'foo',
    ['    issues(assigneeUsernames: "foo", before: $before, after: $after, first: $limit) {'],
  ],
  [
    'current user with username (case insensitive)',
    'assignee = cUrrEnTUsEr()',
    'foo',
    ['    issues(assigneeUsernames: "foo", before: $before, after: $after, first: $limit) {'],
  ],
  [
    'current user without username',
    'assignee = currentUser()',
    '',
    [
      // This is the correct output, the one in Rust was incorrect
      '    issues(assigneeId: "NONE", before: $before, after: $after, first: $limit) {',
    ],
  ],
  [
    'current user with other filters',
    "assignee = currentUser() and label = 'devops::plan'",
    'bar',
    [
      '    issues(',
      '      assigneeUsernames: "bar"',
      '      labelName: "devops::plan"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'current user with special chars',
    'assignee = currentUser()',
    'user.name-123',
    [
      '    issues(',
      '      assigneeUsernames: "user.name-123"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'current user in array',
    'assignee = ("foo",currentUser())',
    'bar',
    [
      '    issues(',
      '      assigneeUsernames: ["foo", "bar"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
])('%s', (_, query, username, expected) => {
  expect(
    compileWithUserContext(username, `project = "gitlab-org/gitlab" and ${query}`).split('\n'),
  ).toEqual(expect.arrayContaining(expected));
});

test.each([
  [
    'relative date literal days',
    'created > 3d',
    '2023-11-23',
    [
      '    issues(',
      '      createdAfter: "2023-11-26 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'relative date literal weeks',
    'created > 1w',
    '2023-11-23',
    [
      '    issues(',
      '      createdAfter: "2023-11-30 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'relative date literal months',
    'created > 1m',
    '2023-11-23',
    [
      '    issues(',
      '      createdAfter: "2023-12-23 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'relative date literal years',
    'created > 1y',
    '2023-11-23',
    [
      '    issues(',
      '      createdAfter: "2024-11-23 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'relative date literal negative',
    'created > -1w',
    '2023-11-23',
    [
      '    issues(',
      '      createdAfter: "2023-11-16 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'relative date literal multiple months',
    'created > 4m',
    '2023-11-23',
    [
      '    issues(',
      '      createdAfter: "2024-03-23 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'relative date literal months over year',
    'created > 12m',
    '2023-11-23',
    [
      '    issues(',
      '      createdAfter: "2024-11-23 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'relative date literal months outside bounds',
    'created > -2m',
    '2023-04-30',
    [
      '    issues(',
      '      createdAfter: "2023-02-28 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'relative date literal months leap year',
    'created > -2m',
    '2024-04-30',
    [
      '    issues(',
      '      createdAfter: "2024-02-29 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'date literal',
    'created > 2023-11-26',
    '2023-11-23',
    [
      '    issues(',
      '      createdAfter: "2023-11-26 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'date function timezone',
    'created > today()',
    '2024-03-10',
    [
      '    issues(',
      '      createdAfter: "2024-03-10 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
  [
    'relative date literal dst transition',
    'created > 2d',
    '2024-03-09',
    [
      '    issues(',
      '      createdAfter: "2024-03-11 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ],
])('%s', (_, query, date, expected) => {
  expect(
    compileWithTimeFromString(date, `project = "gitlab-org/gitlab" and ${query}`).split('\n'),
  ).toEqual(expect.arrayContaining(expected));
});

test.each`
  date
  ${'20231-11-26'}
  ${'2023-13-26'}
  ${'2023-11-42'}
  ${'2023-05'}
  ${'2023-'}
`('invalid date format: $date', ({ date }) => {
  expect(compileGraphQL(`created > ${date}`)).toContain(
    `Error: Invalid date format \`${date}\`. Expected format: yyyy-mm-dd`,
  );
});
