import { featureFlags } from '../../src/utils/feature_flags';
import { compileWithFields, normalizeGraphQL, randomizeCase } from './test_helpers';

test.each`
  field                     | aliases                | responseContains
  ${'assignee'}             | ${['assignees']}       | ${'widgets { ... on WorkItemWidgetAssignees { type assignees { nodes { id avatarUrl username name webUrl } } } }'}
  ${'label'}                | ${['labels']}          | ${'widgets { ... on WorkItemWidgetLabels { type labels { nodes { id title color textColor } } } }'}
  ${'milestone'}            | ${[]}                  | ${'widgets { ... on WorkItemWidgetMilestone { type milestone { id iid dueDate startDate title webPath } } }'}
  ${'startDate'}            | ${['start']}           | ${'widgets { ... on WorkItemWidgetStartAndDueDate { type startDate } } }'}
  ${'dueDate'}              | ${['due']}             | ${'widgets { ... on WorkItemWidgetStartAndDueDate { type dueDate } } }'}
  ${'timeEstimate'}         | ${[]}                  | ${'widgets { ... on WorkItemWidgetTimeTracking { type timeEstimate } } }'}
  ${'totalTimeSpent'}       | ${[]}                  | ${'widgets { ... on WorkItemWidgetTimeTracking { type totalTimeSpent } } }'}
  ${'healthStatus'}         | ${['health']}          | ${'widgets { ... on WorkItemWidgetHealthStatus { type healthStatus } } }'}
  ${'iteration'}            | ${[]}                  | ${'widgets { ... on WorkItemWidgetIteration { type iteration { id iid startDate dueDate title webUrl iterationCadence { id title } } } }'}
  ${'weight'}               | ${[]}                  | ${'widgets { ... on WorkItemWidgetWeight { type weight } } }'}
  ${'progress'}             | ${[]}                  | ${'widgets { ... on WorkItemWidgetProgress { type currentValue endValue progress startValue updatedAt } } }'}
  ${'color'}                | ${[]}                  | ${'widgets { ... on WorkItemWidgetColor { type color textColor } } }'}
  ${'epic'}                 | ${['parent']}          | ${'widgets { ... on WorkItemWidgetHierarchy { type parent { id iid reference state title webUrl } } } }'}
  ${'taskCompletionStatus'} | ${[]}                  | ${'widgets { ... on WorkItemWidgetDescription { type taskCompletionStatus { completedCount count } } } }'}
  ${'lastComment'}          | ${[]}                  | ${'widgets { ... on WorkItemWidgetNotes { type lastComment: notes(filter: ONLY_COMMENTS, last: 1) { nodes { bodyHtml } } } } }'}
  ${'status'}               | ${[]}                  | ${'widgets { ... on WorkItemWidgetStatus { type status { category color description iconName name } } } }'}
  ${'author'}               | ${[]}                  | ${'author { id avatarUrl username name webUrl }'}
  ${'closedAt'}             | ${['closed']}          | ${'closedAt'}
  ${'confidential'}         | ${[]}                  | ${'confidential'}
  ${'createdAt'}            | ${['created']}         | ${'createdAt'}
  ${'description'}          | ${['descriptionHtml']} | ${'descriptionHtml'}
  ${'id'}                   | ${[]}                  | ${'id'}
  ${'iid'}                  | ${[]}                  | ${'iid'}
  ${'state'}                | ${[]}                  | ${'state'}
  ${'title'}                | ${[]}                  | ${'title'}
  ${'updatedAt'}            | ${['updated']}         | ${'updatedAt'}
  ${'webUrl'}               | ${[]}                  | ${'webUrl'}
  ${'type'}                 | ${['workItemType']}    | ${'workItemType { name iconName }'}
  ${'reference'}            | ${[]}                  | ${'reference'}
`(
  'field selection for $field in issues and epics (work items)',
  ({ field, aliases, responseContains }) => {
    featureFlags.glqlWorkItems = true;

    for (const f of [field, ...aliases]) {
      expect(
        normalizeGraphQL(
          compileWithFields(
            `group = "gitlab-org" and type = Issue`,
            // fuzz the field name to make sure we're not case-sensitive
            randomizeCase(f),
          ),
        ),
      ).toContain(responseContains);
    }

    for (const f of [field, ...aliases]) {
      expect(
        normalizeGraphQL(
          compileWithFields(
            `group = "gitlab-org" and type = Epic`,
            // fuzz the field name to make sure we're not case-sensitive
            randomizeCase(f),
          ),
        ),
      ).toContain(responseContains);
    }
  },
);

test.each`
  field                     | aliases                | responseContains
  ${'assignee'}             | ${['assignees']}       | ${'assignees { nodes { id avatarUrl username name webUrl } }'}
  ${'label'}                | ${['labels']}          | ${'labels { nodes { id title color textColor } }'}
  ${'milestone'}            | ${[]}                  | ${'milestone { id iid dueDate startDate title webPath }'}
  ${'dueDate'}              | ${['due']}             | ${'dueDate'}
  ${'timeEstimate'}         | ${[]}                  | ${'timeEstimate'}
  ${'totalTimeSpent'}       | ${[]}                  | ${'totalTimeSpent'}
  ${'healthStatus'}         | ${['health']}          | ${'healthStatus'}
  ${'iteration'}            | ${[]}                  | ${'iteration { id iid startDate dueDate title webUrl iterationCadence { id title } }'}
  ${'weight'}               | ${[]}                  | ${'weight'}
  ${'epic'}                 | ${[]}                  | ${'epic { id iid reference state title webUrl }'}
  ${'taskCompletionStatus'} | ${[]}                  | ${'taskCompletionStatus { completedCount count }'}
  ${'lastComment'}          | ${[]}                  | ${'lastComment: notes(filter: ONLY_COMMENTS, last: 1) { nodes { bodyHtml } }'}
  ${'status'}               | ${[]}                  | ${'status { category color description iconName name }'}
  ${'author'}               | ${[]}                  | ${'author { id avatarUrl username name webUrl }'}
  ${'closedAt'}             | ${['closed']}          | ${'closedAt'}
  ${'confidential'}         | ${[]}                  | ${'confidential'}
  ${'createdAt'}            | ${['created']}         | ${'createdAt'}
  ${'description'}          | ${['descriptionHtml']} | ${'descriptionHtml'}
  ${'id'}                   | ${[]}                  | ${'id'}
  ${'iid'}                  | ${[]}                  | ${'iid'}
  ${'state'}                | ${[]}                  | ${'state'}
  ${'title'}                | ${[]}                  | ${'title'}
  ${'updatedAt'}            | ${['updated']}         | ${'updatedAt'}
  ${'webUrl'}               | ${[]}                  | ${'webUrl'}
  ${'type'}                 | ${[]}                  | ${'type'}
  ${'reference'}            | ${[]}                  | ${'reference'}
`(
  'field selection for $field in issues (not work items)',
  ({ field, aliases, responseContains }) => {
    for (const f of [field, ...aliases]) {
      expect(
        normalizeGraphQL(
          compileWithFields(
            `group = "gitlab-org" and type = Issue`,
            // fuzz the field name to make sure we're not case-sensitive
            randomizeCase(f),
          ),
        ),
      ).toContain(responseContains);
    }
  },
);

test.each`
  field                     | aliases                        | responseContains
  ${'assignee'}             | ${['assignees']}               | ${'assignees { nodes { id avatarUrl username name webUrl } }'}
  ${'approvedBy'}           | ${['approver', 'approvers']}   | ${'approvedBy { nodes { id avatarUrl username name webUrl } }'}
  ${'reviewers'}            | ${['reviewers', 'reviewedBy']} | ${'reviewers { nodes { id avatarUrl username name webUrl } }'}
  ${'mergeUser'}            | ${['mergedBy']}                | ${'mergeUser { id avatarUrl username name webUrl }'}
  ${'sourceProject'}        | ${[]}                          | ${'sourceProject { fullPath webUrl nameWithNamespace }'}
  ${'targetProject'}        | ${[]}                          | ${'targetProject { fullPath webUrl nameWithNamespace }'}
  ${'label'}                | ${['labels']}                  | ${'labels { nodes { id title color textColor } }'}
  ${'milestone'}            | ${[]}                          | ${'milestone { id iid dueDate startDate title webPath }'}
  ${'timeEstimate'}         | ${[]}                          | ${'timeEstimate'}
  ${'totalTimeSpent'}       | ${[]}                          | ${'totalTimeSpent'}
  ${'taskCompletionStatus'} | ${[]}                          | ${'taskCompletionStatus { completedCount count }'}
  ${'lastComment'}          | ${[]}                          | ${'lastComment: notes(filter: ONLY_COMMENTS, last: 1) { nodes { bodyHtml } }'}
  ${'author'}               | ${[]}                          | ${'author { id avatarUrl username name webUrl }'}
  ${'closedAt'}             | ${['closed']}                  | ${'closedAt'}
  ${'createdAt'}            | ${['created']}                 | ${'createdAt'}
  ${'description'}          | ${['descriptionHtml']}         | ${'descriptionHtml'}
  ${'id'}                   | ${[]}                          | ${'id'}
  ${'iid'}                  | ${[]}                          | ${'iid'}
  ${'state'}                | ${[]}                          | ${'state'}
  ${'title'}                | ${[]}                          | ${'title'}
  ${'updatedAt'}            | ${['updated']}                 | ${'updatedAt'}
  ${'webUrl'}               | ${[]}                          | ${'webUrl'}
  ${'reference'}            | ${[]}                          | ${'reference'}
  ${'sourceBranch'}         | ${['sourcebranches']}          | ${'sourceBranch'}
  ${'targetBranch'}         | ${['targetbranches']}          | ${'targetBranch'}
  ${'draft'}                | ${[]}                          | ${'draft'}
`('field selection for $field in merge requests', ({ field, aliases, responseContains }) => {
  for (const f of [field, ...aliases]) {
    expect(
      normalizeGraphQL(
        compileWithFields(
          `group = "gitlab-org" and type = MergeRequest`,
          // fuzz the field name to make sure we're not case-sensitive
          randomizeCase(f),
        ),
      ),
    ).toContain(responseContains);
  }
});
