import { compile } from '../../src/lib';
import { featureFlags } from '../../src/utils/feature_flags';
import { compileGraphQL, compileWithGroupContext } from './test_helpers';

function testCase(query: string, expectedIssues: string[], expectedWorkItems: string[]) {
  featureFlags.glqlWorkItems = false;

  expect(compileGraphQL(`project = "gitlab-org/gitlab" and ${query}`).split('\n')).toEqual(
    expect.arrayContaining(expectedIssues),
  );

  featureFlags.glqlWorkItems = true;

  expect(compileGraphQL(`project = "gitlab-org/gitlab" and ${query}`).split('\n')).toEqual(
    expect.arrayContaining(expectedWorkItems),
  );
}

test('epic equals null', () => {
  testCase(
    'epic = null',
    [
      '    issues(',
      '      epicWildcardId: NONE',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeSubepics: true',
      '    ) {',
    ],
    [
      // Not yet supported for work items
      'Error: `epic` cannot be compared with `null`. Supported value types: `String`, `Number`, `Epic` (example: `&123` or `group-path&123`), `List`.',
    ],
  );
});

test('epic equals none token', () => {
  testCase(
    'epic = NoNe',
    [
      '    issues(',
      '      epicWildcardId: NONE',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeSubepics: true',
      '    ) {',
    ],
    [
      // Not yet supported for work items
      'Error: `epic` cannot be compared with `NoNe`. Supported value types: `String`, `Number`, `Epic` (example: `&123` or `group-path&123`), `List`.',
    ],
  );
});

test('epic equals any token', () => {
  testCase(
    'epic = Any',
    [
      '    issues(',
      '      epicWildcardId: ANY',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeSubepics: true',
      '    ) {',
    ],
    [
      // Not yet supported for work items
      'Error: `epic` cannot be compared with `Any`. Supported value types: `String`, `Number`, `Epic` (example: `&123` or `group-path&123`), `List`.',
    ],
  );
});

test.each`
  valueType   | value
  ${'number'} | ${2}
  ${'string'} | ${'2'}
`('epic equals $valueType', ({ value }) => {
  const compiled = compile(`group = "foo" and epic = ${value}`, {
    fields: 'id',
  });

  expect(compiled.variables['epicId1']!.type).toEqual('String');
  expect(compiled.output.split('\n')).toEqual(
    expect.arrayContaining([
      `query GLQL($epicId1: String, $before: String, $after: String, $limit: Int) {`,
      `  group(fullPath: "foo") {`,
      `    issues(`,
      `      epicId: $epicId1`,
      `      before: $before`,
      `      after: $after`,
      `      first: $limit`,
      `      includeSubgroups: true`,
      `      includeSubepics: true`,
      `    ) {`,
    ]),
  );

  expect(compiled.variables['epicId1']!.value).toMatchInlineSnapshot(`
"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "foo") {
    epics(
      iids: "2"
      includeDescendantGroups: false
      before: $before
      after: $after
      first: $limit
    ) {
      nodes {
        id
      }
    }
  }
}
"
`);
});

test('epic equals group and id', () => {
  const compiled = compile("project = 'gitlab-org/gitlab' and epic = 'gitlab-org&123'", {
    fields: 'id',
  });

  expect(compiled.variables['epicId1']!.type).toEqual('String');
  expect(compiled.output.split('\n')).toEqual(
    expect.arrayContaining([
      `query GLQL($epicId1: String, $before: String, $after: String, $limit: Int) {`,
      `  project(fullPath: "gitlab-org/gitlab") {`,
      `    issues(`,
      `      epicId: $epicId1`,
      `      before: $before`,
      `      after: $after`,
      `      first: $limit`,
      `      includeSubepics: true`,
      `    ) {`,
    ]),
  );

  expect(compiled.variables['epicId1']!.value).toMatchInlineSnapshot(`
"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-org") {
    epics(
      iids: "123"
      includeDescendantGroups: false
      before: $before
      after: $after
      first: $limit
    ) {
      nodes {
        id
      }
    }
  }
}
"
`);
});

test('epic equals string with work items', () => {
  featureFlags.glqlWorkItems = true;

  const compiled = compile("project = 'gitlab-org/gitlab' and epic = 'gitlab-org&1'", {
    fields: 'status',
  });

  expect(compiled.variables['epicId1']!.type).toEqual('WorkItemID!');
  expect(compiled.output.split('\n')).toEqual(
    expect.arrayContaining([
      `query GLQL($epicId1: WorkItemID!, $before: String, $after: String, $limit: Int) {`,
      `  project(fullPath: "gitlab-org/gitlab") {`,
      `    workItems(`,
      `      parentIds: [$epicId1]`,
      `      before: $before`,
      `      after: $after`,
      `      first: $limit`,
      `      includeDescendantWorkItems: true`,
      `    ) {`,
    ]),
  );

  expect(compiled.variables['epicId1']!.value).toMatchInlineSnapshot(`
"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-org") {
    workItems(
      types: EPIC
      iids: "1"
      includeDescendants: false
      before: $before
      after: $after
      first: $limit
      excludeProjects: true
    ) {
      nodes {
        id
      }
    }
  }
}
"
`);
});

test('parent equals epic string with work items', () => {
  featureFlags.glqlWorkItems = true;

  const compiled = compile("project = 'gitlab-org/gitlab' and parent = 'gitlab-org&1'", {
    fields: 'status',
  });

  expect(compiled.variables['parentId1']!.type).toEqual('WorkItemID!');
  expect(compiled.output.split('\n')).toEqual(
    expect.arrayContaining([
      `query GLQL($parentId1: WorkItemID!, $before: String, $after: String, $limit: Int) {`,
    ]),
  );

  expect(compiled.variables['parentId1']!.value).toMatchInlineSnapshot(`
"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-org") {
    workItems(
      iids: "1"
      types: EPIC
      includeDescendants: false
      before: $before
      after: $after
      first: $limit
      excludeProjects: true
    ) {
      nodes {
        id
      }
    }
  }
}
"
`);
});

test('parent equals issue string with work items', () => {
  featureFlags.glqlWorkItems = true;

  const compiled = compile("project = 'gitlab-org/gitlab' and parent = 'gitlab-org#1'", {
    fields: 'status',
  });

  expect(compiled.variables['parentId1']!.type).toEqual('WorkItemID!');
  expect(compiled.output.split('\n')).toEqual(
    expect.arrayContaining([
      `query GLQL($parentId1: WorkItemID!, $before: String, $after: String, $limit: Int) {`,
    ]),
  );

  expect(compiled.variables['parentId1']!.value).toMatchInlineSnapshot(`
"query GLQL($before: String, $after: String, $limit: Int) {
  project(fullPath: "gitlab-org") {
    workItems(iids: "1", before: $before, after: $after, first: $limit) {
      nodes {
        id
      }
    }
  }
}
"
`);
});

test('epic in list with work items', () => {
  featureFlags.glqlWorkItems = true;

  const compiled = compile(
    "project = 'gitlab-org/gitlab' and epic in (gitlab-test&1, gitlab-test&2, gitlab-com&3, &4)",
    { fields: 'id' },
  );

  expect(compiled.output.split('\n')).toEqual(
    expect.arrayContaining([
      `query GLQL($epicId1: WorkItemID!, $epicId2: WorkItemID!, $epicId3: WorkItemID!, $epicId4: WorkItemID!, $before: String, $after: String, $limit: Int) {`,
      `  project(fullPath: "gitlab-org/gitlab") {`,
      `    workItems(`,
      `      parentIds: [$epicId1, $epicId2, $epicId3, $epicId4]`,
      `      before: $before`,
      `      after: $after`,
      `      first: $limit`,
      `      includeDescendantWorkItems: true`,
      `    ) {`,
    ]),
  );

  expect(Object.values(compiled.variables).slice(0, 4)).toMatchInlineSnapshot(`
[
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-test") {
    workItems(
      types: EPIC
      iids: "1"
      includeDescendants: false
      before: $before
      after: $after
      first: $limit
      excludeProjects: true
    ) {
      nodes {
        id
      }
    }
  }
}
",
  },
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-test") {
    workItems(
      types: EPIC
      iids: "2"
      includeDescendants: false
      before: $before
      after: $after
      first: $limit
      excludeProjects: true
    ) {
      nodes {
        id
      }
    }
  }
}
",
  },
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-com") {
    workItems(
      types: EPIC
      iids: "3"
      includeDescendants: false
      before: $before
      after: $after
      first: $limit
      excludeProjects: true
    ) {
      nodes {
        id
      }
    }
  }
}
",
  },
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-org") {
    workItems(
      types: EPIC
      iids: "4"
      includeDescendants: false
      before: $before
      after: $after
      first: $limit
      excludeProjects: true
    ) {
      nodes {
        id
      }
    }
  }
}
",
  },
]
`);
});

test('parent in list with epic work items', () => {
  featureFlags.glqlWorkItems = true;

  const compiled = compile(
    "project = 'gitlab-org/gitlab' and parent in (gitlab-test&1, gitlab-test&2, gitlab-com&3, &4)",
    { fields: 'id' },
  );

  expect(compiled.output.split('\n')).toEqual(
    expect.arrayContaining([
      `query GLQL($parentId1: WorkItemID!, $parentId2: WorkItemID!, $parentId3: WorkItemID!, $parentId4: WorkItemID!, $before: String, $after: String, $limit: Int) {`,
      `  project(fullPath: "gitlab-org/gitlab") {`,
      `    workItems(`,
      `      parentIds: [$parentId1, $parentId2, $parentId3, $parentId4]`,
      `      before: $before`,
      `      after: $after`,
      `      first: $limit`,
      `      includeDescendantWorkItems: true`,
      `    ) {`,
    ]),
  );

  expect(Object.values(compiled.variables).slice(0, 4)).toMatchInlineSnapshot(`
[
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-test") {
    workItems(
      iids: "1"
      types: EPIC
      includeDescendants: false
      before: $before
      after: $after
      first: $limit
      excludeProjects: true
    ) {
      nodes {
        id
      }
    }
  }
}
",
  },
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-test") {
    workItems(
      iids: "2"
      types: EPIC
      includeDescendants: false
      before: $before
      after: $after
      first: $limit
      excludeProjects: true
    ) {
      nodes {
        id
      }
    }
  }
}
",
  },
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-com") {
    workItems(
      iids: "3"
      types: EPIC
      includeDescendants: false
      before: $before
      after: $after
      first: $limit
      excludeProjects: true
    ) {
      nodes {
        id
      }
    }
  }
}
",
  },
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-org") {
    workItems(
      iids: "4"
      types: EPIC
      includeDescendants: false
      before: $before
      after: $after
      first: $limit
      excludeProjects: true
    ) {
      nodes {
        id
      }
    }
  }
}
",
  },
]
`);
});

test('parent in list with issue work items', () => {
  featureFlags.glqlWorkItems = true;

  const compiled = compile(
    "project = 'gitlab-org/gitlab' and parent in (gitlab-org/gitlab-test#1, gitlab-org/gitlab-test#2, gitlab-com/some-project#3, #4)",
    { fields: 'id' },
  );

  expect(compiled.output.split('\n')).toEqual(
    expect.arrayContaining([
      `query GLQL($parentId1: WorkItemID!, $parentId2: WorkItemID!, $parentId3: WorkItemID!, $parentId4: WorkItemID!, $before: String, $after: String, $limit: Int) {`,
      `  project(fullPath: "gitlab-org/gitlab") {`,
      `    workItems(`,
      `      parentIds: [$parentId1, $parentId2, $parentId3, $parentId4]`,
      `      before: $before`,
      `      after: $after`,
      `      first: $limit`,
      `      includeDescendantWorkItems: true`,
      `    ) {`,
    ]),
  );

  expect(Object.values(compiled.variables).slice(0, 4)).toMatchInlineSnapshot(`
[
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  project(fullPath: "gitlab-org/gitlab-test") {
    workItems(iids: "1", before: $before, after: $after, first: $limit) {
      nodes {
        id
      }
    }
  }
}
",
  },
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  project(fullPath: "gitlab-org/gitlab-test") {
    workItems(iids: "2", before: $before, after: $after, first: $limit) {
      nodes {
        id
      }
    }
  }
}
",
  },
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  project(fullPath: "gitlab-com/some-project") {
    workItems(iids: "3", before: $before, after: $after, first: $limit) {
      nodes {
        id
      }
    }
  }
}
",
  },
  {
    "type": "WorkItemID!",
    "value": "query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-org") {
    workItems(
      iids: "4"
      before: $before
      after: $after
      first: $limit
      includeDescendants: true
    ) {
      nodes {
        id
      }
    }
  }
}
",
  },
]
`);
});

test.each`
  featureFlagValue
  ${true}
  ${false}
`(
  'type equals epic with feature flag glqlWorkItems = $featureFlagValue',
  ({ featureFlagValue }) => {
    featureFlags.glqlWorkItems = featureFlagValue;

    expect(compileWithGroupContext('gitlab-org', 'type = Epic').split('\n')).toEqual(
      expect.arrayContaining([
        '    workItems(',
        '      types: EPIC',
        '      before: $before',
        '      after: $after',
        '      first: $limit',
        '      includeDescendants: true',
        '      excludeProjects: true',
        '    ) {',
      ]),
    );
  },
);

test('type in issue and epic', () => {
  expect(
    // also testing case insensitivity
    compileWithGroupContext('gitlab-org', 'type in (IsSuE, ePiC)'),
  ).toEqual(
    'Error: Type `ePiC` cannot be combined with other types (`IsSuE`) in the same query. Try using `type = ePiC` instead.',
  );

  featureFlags.glqlWorkItems = true;

  expect(compileWithGroupContext('gitlab-org', 'type in (Issue, Epic)').split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      types: [ISSUE, EPIC]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '    ) {',
    ]),
  );
});

test('parent field used without work items feature flag', () => {
  expect(compileWithGroupContext('gitlab-org', 'parent = "gitlab-org&1"')).toEqual(
    'Error: `parent` is not a recognized field for work items.',
  );
});
