import { featureFlags } from '../../src/utils/feature_flags';
import { compileGraphQL, compileWithTimeFromString } from './test_helpers';

function testCase(input: string, expectedIssues: string[], expectedWorkItems: string[]) {
  featureFlags.glqlWorkItems = false;

  expect(compileGraphQL(`project = "gitlab-org/gitlab" and ${input}`).split('\n')).toEqual(
    expect.arrayContaining(expectedIssues),
  );

  featureFlags.glqlWorkItems = true;

  expect(compileGraphQL(`project = "gitlab-org/gitlab" and ${input}`).split('\n')).toEqual(
    expect.arrayContaining(expectedWorkItems),
  );
}

function testCaseRelative(
  referenceTime: string,
  input: string,
  expectedIssues: string[],
  expectedWorkItems: string[],
) {
  featureFlags.glqlWorkItems = false;

  expect(
    compileWithTimeFromString(referenceTime, `project = "gitlab-org/gitlab" and ${input}`).split(
      '\n',
    ),
  ).toEqual(expect.arrayContaining(expectedIssues));

  featureFlags.glqlWorkItems = true;

  expect(
    compileWithTimeFromString(referenceTime, `project = "gitlab-org/gitlab" and ${input}`).split(
      '\n',
    ),
  ).toEqual(expect.arrayContaining(expectedWorkItems));
}

test('due gt absolute date', () => {
  testCase(
    'due > 2024-02-01',
    [
      '    issues(',
      '      dueAfter: "2024-02-01 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueAfter: "2024-02-01 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due gte absolute date', () => {
  testCase(
    'due >= 2024-02-01',
    [
      '    issues(',
      '      dueAfter: "2024-02-01 00:00"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueAfter: "2024-02-01 00:00"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due lt absolute date', () => {
  testCase(
    'due < 2024-02-01',
    [
      '    issues(',
      '      dueBefore: "2024-02-01 00:00"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueBefore: "2024-02-01 00:00"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due lte absolute date', () => {
  testCase(
    'due <= 2024-02-01',
    [
      '    issues(',
      '      dueBefore: "2024-02-01 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueBefore: "2024-02-01 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due lt absolute date string', () => {
  testCase(
    'due < "2024-02-01"',
    [
      '    issues(',
      '      dueBefore: "2024-02-01 00:00"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueBefore: "2024-02-01 00:00"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due lte absolute date string', () => {
  testCase(
    'due <= "2024-02-01"',
    [
      '    issues(',
      '      dueBefore: "2024-02-01 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueBefore: "2024-02-01 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due eq absolute date string', () => {
  testCase(
    'due = "2024-02-01"',
    [
      '    issues(',
      '      dueAfter: "2024-02-01 00:00"',
      '      dueBefore: "2024-02-01 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueAfter: "2024-02-01 00:00"',
      '      dueBefore: "2024-02-01 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due gt relative date days', () => {
  testCaseRelative(
    '2024-11-27',
    'due > -3d',
    [
      '    issues(',
      '      dueAfter: "2024-11-24 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueAfter: "2024-11-24 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due lte relative date months', () => {
  testCaseRelative(
    '2024-11-27',
    'due <= 4m',
    [
      '    issues(',
      '      dueBefore: "2025-03-27 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueBefore: "2025-03-27 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due eq relative date years', () => {
  testCaseRelative(
    '2024-11-27',
    'due = -1y',
    [
      '    issues(',
      '      dueAfter: "2023-11-27 00:00"',
      '      dueBefore: "2023-11-27 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueAfter: "2023-11-27 00:00"',
      '      dueBefore: "2023-11-27 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due gt relative date weeks', () => {
  testCaseRelative(
    '2024-11-27',
    'due > 2w',
    [
      '    issues(',
      '      dueAfter: "2024-12-11 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueAfter: "2024-12-11 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due lt lte equivalence part 1', () => {
  // `due < -5d` is equivalent to `due <= -6d`
  testCaseRelative(
    '2024-11-27',
    'due < -5d',
    [
      '    issues(',
      '      dueBefore: "2024-11-22 00:00"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueBefore: "2024-11-22 00:00"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );

  testCaseRelative(
    '2024-11-27',
    'due <= -6d',
    [
      '    issues(',
      '      dueBefore: "2024-11-21 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueBefore: "2024-11-21 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due gt gte equivalence', () => {
  // `due > -5d` is equivalent to `due >= -4d`
  testCaseRelative(
    '2024-11-27',
    'due > -5d',
    [
      '    issues(',
      '      dueAfter: "2024-11-22 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueAfter: "2024-11-22 23:59"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );

  testCaseRelative(
    '2024-11-27',
    'due >= -4d',
    [
      '    issues(',
      '      dueAfter: "2024-11-23 00:00"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      dueAfter: "2024-11-23 00:00"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  );
});

test('due invalid operator', () => {
  testCase(
    'due != 2024-02-01',
    [
      'Error: `due` does not support the not equals (`!=`) operator. Supported operators: equals (`=`), greater than (`>`), greater than or equal to (`>=`), less than (`<`), less than or equal to (`<=`).',
    ],
    [
      'Error: `due` does not support the not equals (`!=`) operator. Supported operators: equals (`=`), greater than (`>`), greater than or equal to (`>=`), less than (`<`), less than or equal to (`<=`).',
    ],
  );
});

test('due invalid date format', () => {
  testCase(
    'due = "random string"',
    ['Error: `due` cannot be compared with `"random string"`. Supported value types: `Date`.'],
    ['Error: `due` cannot be compared with `"random string"`. Supported value types: `Date`.'],
  );
});

test('due invalid number value', () => {
  testCase(
    'due > 12345',
    ['Error: `due` cannot be compared with `12345`. Supported value types: `Date`.'],
    ['Error: `due` cannot be compared with `12345`. Supported value types: `Date`.'],
  );
});

test('due invalid null value', () => {
  testCase(
    'due < null',
    ['Error: `due` cannot be compared with `null`. Supported value types: `Date`.'],
    ['Error: `due` cannot be compared with `null`. Supported value types: `Date`.'],
  );
});

test('due invalid boolean value', () => {
  testCase(
    'due < true',
    ['Error: `due` cannot be compared with `true`. Supported value types: `Date`.'],
    ['Error: `due` cannot be compared with `true`. Supported value types: `Date`.'],
  );
});
