import { compileGraphQL } from './test_helpers';

function testCase(input: string, expected: string[]) {
  return () => {
    const result = compileGraphQL(`project = "gitlab-org/gitlab" and ${input}`);
    expect(result.split('\n')).toEqual(expect.arrayContaining(expected));
  };
}

test(
  'custom field',
  testCase('customField("Subscription") = "Free"', [
    '    issues(',
    '      before: $before',
    '      after: $after',
    '      first: $limit',
    '      customField: [{customFieldName: "Subscription", selectedOptionValues: "Free"}]',
    '    ) {',
  ]),
);

test(
  'custom field with list',
  testCase('customField("Subscription") = ("Free", "Paid")', [
    '    issues(',
    '      before: $before',
    '      after: $after',
    '      first: $limit',
    '      customField: [{customFieldName: "Subscription", selectedOptionValues: ["Free", "Paid"]}]',
    '    ) {',
  ]),
);

test(
  'custom field combined',
  testCase('customField("Subscription") = "Free" and customField("Subscription") = "Paid"', [
    '    issues(',
    '      before: $before',
    '      after: $after',
    '      first: $limit',
    '      customField: [{customFieldName: "Subscription", selectedOptionValues: ["Free", "Paid"]}]',
    '    ) {',
  ]),
);

test(
  'multiple custom fields',
  testCase('customField("Subscription") = "Free" and customField("Team") = "Green"', [
    '    issues(',
    '      before: $before',
    '      after: $after',
    '      first: $limit',
    '      customField: [{customFieldName: "Subscription", selectedOptionValues: "Free"}, {customFieldName: "Team", selectedOptionValues: "Green"}]',
    '    ) {',
  ]),
);

test(
  'custom field invalid operator',
  testCase('customField("Subscription") != "Free"', [
    'Error: `customField("Subscription")` does not support the not equals (`!=`) operator. Supported operators: equals (`=`).',
  ]),
);

test(
  'custom field invalid syntax 1',
  testCase('customField = "Free"', ['Error: `customField` is not a recognized field.']),
);

test(
  'custom field invalid syntax 2',
  testCase('customField("Subscription", "Team") = "Free"', [
    'Error: Unexpected number of arguments for function: `customField`. Expected: 1, Actual: 2',
  ]),
);

test(
  'invalid function call',
  testCase('fooBar("Subscription") = "Free"', [
    'Error: Unrecognized function: fooBar("Subscription")',
  ]),
);
