import { featureFlags } from '../../src/utils/feature_flags';
import { compileGraphQL } from './test_helpers';

function testCase(
  input: string,
  expectedIssues: string[],
  expectedEpics: string[],
  expectedWorkItems: string[],
) {
  return () => {
    // test issues
    const result = compileGraphQL(`project = "gitlab-org/gitlab" and ${input}`);
    expect(result.split('\n')).toEqual(expect.arrayContaining(expectedIssues));

    // test epics
    const result2 = compileGraphQL(`group = "gitlab-org" and type = Epic and ${input}`);
    expect(result2.split('\n')).toEqual(expect.arrayContaining(expectedEpics));

    // test work items
    featureFlags.glqlWorkItems = true;
    const result3 = compileGraphQL(`project = "gitlab-org/gitlab" and ${input}`);
    expect(result3.split('\n')).toEqual(expect.arrayContaining(expectedWorkItems));
  };
}

test(
  'author equals string',
  testCase(
    'author = "foo"',
    ['    issues(authorUsername: "foo", before: $before, after: $after, first: $limit) {'],
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    ['    workItems(authorUsername: "foo", before: $before, after: $after, first: $limit) {'],
  ),
);

test(
  'author equals username reference',
  testCase(
    'author = @foo',
    ['    issues(authorUsername: "foo", before: $before, after: $after, first: $limit) {'],
    [
      '    workItems(',
      '      types: EPIC',
      '      authorUsername: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    ['    workItems(authorUsername: "foo", before: $before, after: $after, first: $limit) {'],
  ),
);

test(
  'author equals array',
  testCase(
    'author = ("foo", "bar")',
    [
      'Error: `author` does not support the equals (`=`) operator for `("foo", "bar")`. Supported operators: is one of (`in`), not equals (`!=`).',
    ],
    [
      'Error: `author` does not support the equals (`=`) operator for `("foo", "bar")`. Supported operators: is one of (`in`), not equals (`!=`).',
    ],
    [
      'Error: `author` does not support the equals (`=`) operator for `("foo", "bar")`. Supported operators: is one of (`in`), not equals (`!=`).',
    ],
  ),
);

test(
  'author in array',
  testCase(
    'author in ("foo", "bar")',
    [
      '    issues(',
      '      or: {authorUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      or: {authorUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      or: {authorUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ),
);

test(
  'author in array of username references',
  testCase(
    'author in (@foo, @bar)',
    [
      '    issues(',
      '      or: {authorUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      or: {authorUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      or: {authorUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ),
);

test(
  'author in mixed array',
  testCase(
    'author in ("foo", @bar)',
    [
      '    issues(',
      '      or: {authorUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      or: {authorUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      or: {authorUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ),
);

test(
  'author not equal to string',
  testCase(
    'author != "foo"',
    [
      '    issues(',
      '      not: {authorUsername: "foo"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      not: {authorUsername: "foo"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      not: {authorUsername: "foo"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ),
);

test(
  'author not equal to array',
  testCase(
    'author != ("foo", "bar")',
    [
      '    issues(',
      '      not: {authorUsername: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
    [
      '    workItems(',
      '      types: EPIC',
      '      not: {authorUsername: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '      includeDescendants: true',
      '      excludeProjects: true',
      '    ) {',
    ],
    [
      '    workItems(',
      '      not: {authorUsername: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ],
  ),
);
