import { featureFlags } from '../../src/utils/feature_flags';
import { compileGraphQL } from './test_helpers';

test('assignee equals string', () => {
  const result = compileGraphQL('project = "gitlab-org/gitlab" and assignee = "foo"');
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(assigneeUsernames: "foo", before: $before, after: $after, first: $limit) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;
  const result2 = compileGraphQL('project = "gitlab-org/gitlab" and assignee = "foo"');

  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee equals username reference', () => {
  const result = compileGraphQL('project = "gitlab-org/gitlab" and assignee = @foo');
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(assigneeUsernames: "foo", before: $before, after: $after, first: $limit) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;
  const result2 = compileGraphQL('project = "gitlab-org/gitlab" and assignee = @foo');
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: "foo"',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee equals array', () => {
  const result = compileGraphQL('project = "gitlab-org/gitlab" and assignee = ("foo", "bar")');

  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: ["foo", "bar"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;
  const result2 = compileGraphQL('project = "gitlab-org/gitlab" and assignee = ("foo", "bar")');
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: ["foo", "bar"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee equals array of username references', () => {
  const result = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = (@.foo, @bar-baz, @qux_quux)',
  );

  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: [".foo", "bar-baz", "qux_quux"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = (@.foo, @bar-baz, @qux_quux)',
  );
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: [".foo", "bar-baz", "qux_quux"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee equals mixed array', () => {
  const result = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = ("foo", @bar, "baz")',
  );
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: ["foo", "bar", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = ("foo", @bar, "baz")',
  );
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: ["foo", "bar", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee equals empty array', () => {
  const result = compileGraphQL('project = "gitlab-org/gitlab" and assignee=()');
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(assigneeWildcardId: NONE, before: $before, after: $after, first: $limit) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL('project = "gitlab-org/gitlab" and assignee=()');
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeWildcardId: NONE',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee not equals empty array', () => {
  const result = compileGraphQL('project = "gitlab-org/gitlab" and assignee != ()');
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(assigneeWildcardId: ANY, before: $before, after: $after, first: $limit) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL('project = "gitlab-org/gitlab" and assignee != ()');
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeWildcardId: ANY',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee in', () => {
  const result = compileGraphQL('project = "gitlab-org/gitlab" and assignee in ("foo", "bar")');
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      or: {assigneeUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL('project = "gitlab-org/gitlab" and assignee in ("foo", "bar")');
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      or: {assigneeUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee multiple operator', () => {
  const result = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee in ("foo", "bar") and assignee = "baz"',
  );
  expect(result.split('\n').slice(2, 9)).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: "baz"',
      '      or: {assigneeUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee in ("foo", "bar") and assignee = "baz"',
  );
  expect(result2.split('\n').slice(2, 9)).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: "baz"',
      '      or: {assigneeUsernames: ["foo", "bar"]}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee two equals string', () => {
  const result = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = "bar"',
  );
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: ["foo", "bar"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = "bar"',
  );
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: ["foo", "bar"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee multiple equals string', () => {
  const result = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = "bar" and assignee="boo"',
  );
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: ["foo", "bar", "boo"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = "bar" and assignee="boo"',
  );
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: ["foo", "bar", "boo"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee equals string and array', () => {
  const result = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = ("bar", "boo")',
  );
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: ["foo", "bar", "boo"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = ("bar", "boo")',
  );
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: ["foo", "bar", "boo"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee multiple equals string and array', () => {
  const result = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = ("bar", "boo") and assignee="baz"',
  );
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: ["foo", "bar", "boo", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = ("bar", "boo") and assignee="baz"',
  );
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: ["foo", "bar", "boo", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee multiple equals array', () => {
  const result = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = ("foo", "bar") and assignee = ("boo", "baz")',
  );
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: ["foo", "bar", "boo", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = ("foo", "bar") and assignee = ("boo", "baz")',
  );
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: ["foo", "bar", "boo", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee not equals', () => {
  const result = compileGraphQL('project = "gitlab-org/gitlab" and assignee != "foo"');
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      not: {assigneeUsernames: "foo"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result2 = compileGraphQL('project = "gitlab-org/gitlab" and assignee != "foo"');
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      not: {assigneeUsernames: "foo"}',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('assignee with null', () => {
  const result = compileGraphQL('project = "gitlab-org/gitlab" and assignee = null');
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(assigneeId: "NONE", before: $before, after: $after, first: $limit) {',
    ]),
  );
});

test('assignee with id', () => {
  const result = compileGraphQL('project = "gitlab-org/gitlab" and assignee=123');
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(assigneeId: "123", before: $before, after: $after, first: $limit) {',
    ]),
  );
});

test('assignee equals milestone reference', () => {
  const result = compileGraphQL('project = "gitlab-org/gitlab" and assignee = %Backlog');
  expect(result).toEqual(
    'Error: `assignee` cannot be compared with `%Backlog`. Supported value types: `String`, `Username` (example: `@username`), `List`, `Number`, `Nullable` (`null`, `none`, `any`).',
  );
});

test('assignee equals mixed references', () => {
  const result = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = @foo and assignee = (@bar, @baz)',
  );
  expect(result.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: ["foo", "bar", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  const result2 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = @foo and assignee = @bar',
  );
  expect(result2.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: ["foo", "bar"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  const result3 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = (@bar, @baz)',
  );
  expect(result3.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: ["foo", "bar", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  const result4 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = ("bar", @baz) and assignee = @quox',
  );
  expect(result4.split('\n')).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      assigneeUsernames: ["foo", "bar", "baz", "quox"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  featureFlags.glqlWorkItems = true;

  const result5 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = ("bar", @baz) and assignee = @quox',
  );
  expect(result5.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: ["foo", "bar", "baz", "quox"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  const result6 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = @foo and assignee = @bar',
  );
  expect(result6.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: ["foo", "bar"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  const result7 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = (@bar, @baz)',
  );
  expect(result7.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: ["foo", "bar", "baz"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  const result8 = compileGraphQL(
    'project = "gitlab-org/gitlab" and assignee = "foo" and assignee = ("bar", @baz) and assignee = @quox',
  );
  expect(result8.split('\n')).toEqual(
    expect.arrayContaining([
      '    workItems(',
      '      assigneeUsernames: ["foo", "bar", "baz", "quox"]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});
