import { compileGraphQL, compileWithTimeFromString } from './test_helpers';

test.each([
  ["foo = 'foo'", 'Error: `foo` is not a recognized field.'],
  [
    'weight > 1',
    'Error: `weight` does not support the greater than (`>`) operator. Supported operators: equals (`=`), not equals (`!=`).',
  ],
  [
    'label IN "fish"',
    'Error: `label` does not support the is one of (`in`) operator for `"fish"`. Supported operators: equals (`=`), not equals (`!=`).',
  ],
  [
    'label IN ()',
    'Error: `label` does not support the is one of (`in`) operator for `()`. Supported operators: equals (`=`), not equals (`!=`).',
  ],
  [
    'label > ()',
    'Error: `label` does not support the greater than (`>`) operator. Supported operators: equals (`=`), not equals (`!=`), is one of (`in`).',
  ],
  [
    'label < ()',
    'Error: `label` does not support the less than (`<`) operator. Supported operators: equals (`=`), not equals (`!=`), is one of (`in`).',
  ],
  [
    "group != 'fish'",
    'Error: `group` does not support the not equals (`!=`) operator. Supported operators: equals (`=`).',
  ],
  [
    "project != 'fish'",
    'Error: `project` does not support the not equals (`!=`) operator. Supported operators: equals (`=`).',
  ],
  [
    "created != 'fish'",
    'Error: `created` does not support the not equals (`!=`) operator. Supported operators: equals (`=`), greater than (`>`), greater than or equal to (`>=`), less than (`<`), less than or equal to (`<=`).',
  ],
  [
    'created = ("fish")',
    'Error: `created` cannot be compared with `("fish")`. Supported value types: `Date`.',
  ],
  [
    "health = 'foo'",
    'Error: `health` cannot be compared with `"foo"`. Supported value types: `StringEnum` (`"on track"`, `"needs attention"`, `"at risk"`), `Nullable` (`null`, `none`, `any`).',
  ],
  [
    'confidential = ()',
    'Error: `confidential` cannot be compared with `()`. Supported value types: `Boolean` (`true`, `false`).',
  ],
  ['milestone = catch("fish")', 'Error: Unrecognized function: catch("fish")'],
  [
    'label IN ("foo") AND label IN ("bar")',
    'Error: Operator is one of (`in`) can only be used once with `label`.',
  ],
  ['foo = ("foo")', 'Error: `foo` is not a recognized field.'],
  ['created = 1', 'Error: `created` cannot be compared with `1`. Supported value types: `Date`.'],
  [
    'opened = "fish"',
    'Error: `opened` cannot be compared with `"fish"`. Supported value types: `Date`.',
  ],
  ['created = 2024/01/01', 'Error: Invalid date format `2024/01/01`. Expected format: yyyy-mm-dd'],
  ['created = 2024.01.01', 'Error: Invalid date format `2024.01.01`. Expected format: yyyy-mm-dd'],
  [
    'created > 1h',
    'Error: Unexpected character `h`. Expected d (day), w (week), m (month), or y (year)',
  ],
  [
    'created > 1s',
    'Error: Unexpected character `s`. Expected d (day), w (week), m (month), or y (year)',
  ],
  ['assignee = @user@name', 'Error: Invalid username reference `@user@name`'],
  ['assignee = @123', 'Error: Invalid username reference `@123`'],
  ['label = (("foo"))', 'Error: Expected valid value near `(("foo"))`'],
])('%s', (query, expected) => {
  expect(compileGraphQL(query)).toBe(expected);
});

// multiple operators on the same field
describe.each`
  field
  ${'created'}
  ${'updated'}
  ${'closed'}
`('field: $field', ({ field }) => {
  test.each`
    operator | operatorDescription
    ${'='}   | ${'equals'}
    ${'>'}   | ${'greater than'}
    ${'>='}  | ${'greater than or equal to'}
    ${'<'}   | ${'less than'}
    ${'<='}  | ${'less than or equal to'}
  `(
    `${field} $operator 2024-01-01 AND ${field} $operator 2024-01-01`,
    ({ operator, operatorDescription }) => {
      expect(
        compileGraphQL(`${field} ${operator} 2024-01-01 AND ${field} ${operator} 2024-01-01`),
      ).toBe(
        `Error: Operator ${operatorDescription} (\`${operator}\`) can only be used once with \`${field}\`.`,
      );
    },
  );
});

describe.each`
  field
  ${'confidential'}
`('field: $field', ({ field }) => {
  test.each`
    operator | operatorDescription
    ${'='}   | ${'equals'}
  `(`${field} $operator true AND ${field} $operator false`, ({ operator, operatorDescription }) => {
    expect(compileGraphQL(`${field} ${operator} true AND ${field} ${operator} false`)).toBe(
      `Error: Operator ${operatorDescription} (\`${operator}\`) can only be used once with \`${field}\`.`,
    );
  });
});

// test valid cases
test.each([
  [
    'milestone in ("fish")',
    ['  issues(milestoneTitle: ["fish"], before: $before, after: $after, first: $limit) {'],
  ],
  [
    'milestone != none',
    ['  issues(milestoneWildcardId: ANY, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'milestone != any',
    ['  issues(milestoneWildcardId: NONE, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'confidential = null',
    ['  issues(confidential: false, before: $before, after: $after, first: $limit) {'],
  ],
  [
    "confidential = 'false'",
    ['  issues(confidential: false, before: $before, after: $after, first: $limit) {'],
  ],
  [
    "confidential = 'faLse'",
    ['  issues(confidential: false, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'confidential = 0',
    ['  issues(confidential: false, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'confidential = 1',
    ['  issues(confidential: true, before: $before, after: $after, first: $limit) {'],
  ],
  [
    "confidential = 'true'",
    ['  issues(confidential: true, before: $before, after: $after, first: $limit) {'],
  ],
  [
    "confidential = 'tRue'",
    ['  issues(confidential: true, before: $before, after: $after, first: $limit) {'],
  ],
  [
    'label = "foo\\"bar"',
    ['  issues(labelName: "foo\\"bar", before: $before, after: $after, first: $limit) {'],
  ],
  [
    "label = ('foo') AND label in ('bar')",
    [
      '  issues(',
      '    labelName: ["foo"]',
      '    or: {labelNames: ["bar"]}',
      '    before: $before',
      '    after: $after',
      '    first: $limit',
      '  ) {',
    ],
  ],
])('%s', (query, expected) => {
  expect(compileGraphQL(query).split('\n')).toEqual(expect.arrayContaining(expected));
});

describe.each`
  field
  ${'created'}
  ${'updated'}
  ${'closed'}
`('field: $field', ({ field }) => {
  test.each([
    [
      `${field} = 2024-01-01`,
      [
        '  issues(',
        `    ${field}After: "2024-01-01 00:00"`,
        `    ${field}Before: "2024-01-01 23:59"`,
        '    before: $before',
        '    after: $after',
        '    first: $limit',
        '  ) {',
      ],
    ],
    [
      `${field} = '2024-01-01'`,
      [
        '  issues(',
        `    ${field}After: "2024-01-01 00:00"`,
        `    ${field}Before: "2024-01-01 23:59"`,
        '    before: $before',
        '    after: $after',
        '    first: $limit',
        '  ) {',
      ],
    ],
  ])('%s', (query, expected) => {
    expect(compileGraphQL(query).split('\n')).toEqual(expect.arrayContaining(expected));
  });

  test.each([
    [
      `${field} = -1d`,
      [
        '  issues(',
        `    ${field}After: "2024-07-11 00:00"`,
        `    ${field}Before: "2024-07-11 23:59"`,
        '    before: $before',
        '    after: $after',
        '    first: $limit',
        '  ) {',
      ],
    ],
  ])('%s', (query, expected) => {
    expect(compileWithTimeFromString('2024-07-12', query).split('\n')).toEqual(
      expect.arrayContaining(expected),
    );
  });
});
