import { Dimension } from '../../src/types/dimension';
import { DimensionFunction } from '../../src/types/dimension_function';
import { DisplayField } from '../../src/types/display_field';
import { ReferenceType } from '../../src/types/reference_type';
import { Value } from '../../src/types/value';
import { TimeSegmentType } from '../../src/utils/date_time/time_segment_type';
import { TimeUnit } from '../../src/utils/date_time/time_unit';
import { compileWithAggregate } from './test_helpers';

describe('Aggregate Tests', () => {
  test('aggregate with dimensions and metrics (count, totalTimeToMerge, meanTimeToMerge)', () => {
    const query =
      "project = 'gitlab-org/gitlab' and type = MergeRequest and merged >= '2025-07-09' and merged <= '2025-08-08'";
    const dimensions = 'timeSegment(1m) on merged';
    const metrics = 'count, totalTimeToMerge, meanTimeToMerge';
    const result = compileWithAggregate(query, dimensions, metrics);

    // Repetition of totalTimeToMerge and count is expected
    // because meanTimeToMerge is derived from totalTimeToMerge and count
    // in the transform step
    expect(result.output).toMatchInlineSnapshot(`
"query GLQL {
  project(fullPath: "gitlab-org/gitlab") {
    glql_s1: mergeRequests(
      mergedAfter: "2025-07-09 00:00"
      mergedBefore: "2025-08-01 00:00"
      first: 0
    ) {
      count
      totalTimeToMerge
      totalTimeToMerge
      count
    }
    glql_s2: mergeRequests(
      mergedAfter: "2025-08-01 00:00"
      mergedBefore: "2025-08-08 23:59"
      first: 0
    ) {
      count
      totalTimeToMerge
      totalTimeToMerge
      count
    }
  }
}
"
`);

    expect(result.groupBy).toEqual([
      new Dimension(
        new DimensionFunction.Time(1, TimeUnit.Month, TimeSegmentType.FromStartOfUnit),
        DisplayField.fromString('merged'),
      ),
    ]);

    expect(result.aggregate).toEqual([
      DisplayField.fromString('count'),
      DisplayField.fromString('totalTimeToMerge'),
      DisplayField.fromString('meanTimeToMerge'),
    ]);
  });

  test('aggregate with dimensions and metrics (only meanTimeToMerge)', () => {
    const query =
      "project = 'gitlab-org/gitlab' and type = MergeRequest and merged >= '2025-07-09' and merged <= '2025-08-08'";
    const dimensions = 'timeSegment(1m) on merged';
    const metrics = 'meanTimeToMerge';
    const result = compileWithAggregate(query, dimensions, metrics);

    // Repetition of totalTimeToMerge and count is expected
    // because meanTimeToMerge is derived from totalTimeToMerge and count
    // in the transform step
    expect(result.output).toMatchInlineSnapshot(`
"query GLQL {
  project(fullPath: "gitlab-org/gitlab") {
    glql_s1: mergeRequests(
      mergedAfter: "2025-07-09 00:00"
      mergedBefore: "2025-08-01 00:00"
      first: 0
    ) {
      totalTimeToMerge
      count
    }
    glql_s2: mergeRequests(
      mergedAfter: "2025-08-01 00:00"
      mergedBefore: "2025-08-08 23:59"
      first: 0
    ) {
      totalTimeToMerge
      count
    }
  }
}
"
`);

    expect(result.groupBy).toEqual([
      new Dimension(
        new DimensionFunction.Time(1, TimeUnit.Month, TimeSegmentType.FromStartOfUnit),
        DisplayField.fromString('merged'),
      ),
    ]);

    expect(result.aggregate).toEqual([DisplayField.fromString('meanTimeToMerge')]);
  });

  test('aggregate with multiple dimensions', () => {
    const query =
      "project = 'gitlab-org/gitlab' and type = MergeRequest and merged >= '2024-08-09' and merged <= '2025-08-08'";
    const dimensions = 'timeSegment(3m) on merged, users(@alice, @bob, @charlie) on author';
    const metrics = 'count, totalTimeToMerge';
    const result = compileWithAggregate(query, dimensions, metrics);

    expect(result.groupBy).toEqual([
      new Dimension(
        new DimensionFunction.Time(3, TimeUnit.Month, TimeSegmentType.FromStartOfUnit),
        DisplayField.fromString('merged'),
      ),
      new Dimension(
        new DimensionFunction.User([
          new Value.Reference(ReferenceType.User, 'alice'),
          new Value.Reference(ReferenceType.User, 'bob'),
          new Value.Reference(ReferenceType.User, 'charlie'),
        ]),
        DisplayField.fromString('author'),
      ),
    ]);

    expect(result.aggregate).toEqual([
      DisplayField.fromString('count'),
      DisplayField.fromString('totalTimeToMerge'),
    ]);

    expect(
      result.output.split('\n').filter((line) => line.includes('glql_s')).length,
      // 4 + 1 time segments * 3 users = 15
    ).toBe(15);

    expect(result.output).toMatchInlineSnapshot(`
"query GLQL {
  project(fullPath: "gitlab-org/gitlab") {
    glql_s1_s6: mergeRequests(
      mergedAfter: "2024-08-09 00:00"
      mergedBefore: "2024-10-01 00:00"
      authorUsername: "alice"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s1_s7: mergeRequests(
      mergedAfter: "2024-08-09 00:00"
      mergedBefore: "2024-10-01 00:00"
      authorUsername: "bob"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s1_s8: mergeRequests(
      mergedAfter: "2024-08-09 00:00"
      mergedBefore: "2024-10-01 00:00"
      authorUsername: "charlie"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s2_s6: mergeRequests(
      mergedAfter: "2024-10-01 00:00"
      mergedBefore: "2025-01-01 00:00"
      authorUsername: "alice"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s2_s7: mergeRequests(
      mergedAfter: "2024-10-01 00:00"
      mergedBefore: "2025-01-01 00:00"
      authorUsername: "bob"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s2_s8: mergeRequests(
      mergedAfter: "2024-10-01 00:00"
      mergedBefore: "2025-01-01 00:00"
      authorUsername: "charlie"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s3_s6: mergeRequests(
      mergedAfter: "2025-01-01 00:00"
      mergedBefore: "2025-04-01 00:00"
      authorUsername: "alice"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s3_s7: mergeRequests(
      mergedAfter: "2025-01-01 00:00"
      mergedBefore: "2025-04-01 00:00"
      authorUsername: "bob"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s3_s8: mergeRequests(
      mergedAfter: "2025-01-01 00:00"
      mergedBefore: "2025-04-01 00:00"
      authorUsername: "charlie"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s4_s6: mergeRequests(
      mergedAfter: "2025-04-01 00:00"
      mergedBefore: "2025-07-01 00:00"
      authorUsername: "alice"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s4_s7: mergeRequests(
      mergedAfter: "2025-04-01 00:00"
      mergedBefore: "2025-07-01 00:00"
      authorUsername: "bob"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s4_s8: mergeRequests(
      mergedAfter: "2025-04-01 00:00"
      mergedBefore: "2025-07-01 00:00"
      authorUsername: "charlie"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s5_s6: mergeRequests(
      mergedAfter: "2025-07-01 00:00"
      mergedBefore: "2025-08-08 23:59"
      authorUsername: "alice"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s5_s7: mergeRequests(
      mergedAfter: "2025-07-01 00:00"
      mergedBefore: "2025-08-08 23:59"
      authorUsername: "bob"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
    glql_s5_s8: mergeRequests(
      mergedAfter: "2025-07-01 00:00"
      mergedBefore: "2025-08-08 23:59"
      authorUsername: "charlie"
      first: 0
    ) {
      count
      totalTimeToMerge
    }
  }
}
"
`);
  });
});
