import { parseDateTime, timeSegment } from '.';
import { TimeSegmentType } from './time_segment_type';
import { TimeUnit } from './time_unit';

export class TimeRange {
  constructor(
    public from: Date,
    public to: Date,
  ) {}

  get duration() {
    return this.to.getTime() - this.from.getTime();
  }

  // Format the time range in the format "from_YYYY_MM_DD_to_YYYY_MM_DD"
  get key() {
    const from = this.from.toISOString().split('T')[0]!.replace(/-/g, '_');
    const to = this.to.toISOString().split('T')[0]!.replace(/-/g, '_');
    return `from_${from}_to_${to}`;
  }

  static fromString(from: string, to: string) {
    return new TimeRange(parseDateTime(from), parseDateTime(to));
  }

  // Parse a key in the format "from_YYYY_MM_DD_to_YYYY_MM_DD"
  static fromKey(key: string) {
    try {
      const [from_part, to_part] = key.split('_to_');
      return TimeRange.fromString(
        from_part!.replace('from_', '').replace(/_/g, '-'),
        to_part!.replace(/_/g, '-'),
      );
    } catch {
      throw new Error(`Invalid TimeRange key: ${key}`);
    }
  }

  segment(quantity: number, unit: TimeUnit, segmentType: TimeSegmentType) {
    return timeSegment(this, quantity, unit, segmentType);
  }

  toString() {
    const { from, to } = this.toJSON();
    return `From ${from} to ${to}`;
  }

  toJSON() {
    return {
      from: this.from.toISOString().split('T')[0],
      to: this.to.toISOString().split('T')[0],
    };
  }
}
