import { TimeRange } from './time_range';
import { TimeSegmentType } from './time_segment_type';
import { TimeUnit } from './time_unit';

export function startOfDay(days: number, time = new Date()): string {
  return new Date(time.getTime() + days * 24 * 60 * 60 * 1000).toISOString().split('T')[0]!;
}

export const parseDateTime = (s: string): Date => {
  // Try parsing as DateTime first (YYYY-MM-DD HH:MM)
  const dateTimeMatch = s.match(/^(\d{4})-(\d{2})-(\d{2})[T\s](\d{2}):(\d{2})$/);
  if (dateTimeMatch) {
    const [, year, month, day, hour, minute] = dateTimeMatch;
    return new Date(
      Date.UTC(
        parseInt(year!, 10),
        parseInt(month!, 10) - 1, // Month is 0-indexed
        parseInt(day!, 10),
        parseInt(hour!, 10),
        parseInt(minute!, 10),
        0,
      ),
    );
  }

  // Try as date only (YYYY-MM-DD)
  const dateMatch = s.match(/^(\d{4})-(\d{2})-(\d{2})$/);
  if (dateMatch) {
    const [, year, month, day] = dateMatch;
    return new Date(
      Date.UTC(
        parseInt(year!, 10),
        parseInt(month!, 10) - 1, // Month is 0-indexed
        parseInt(day!, 10),
        0,
        0,
        0,
      ),
    );
  }

  // Fallback to default Date parsing
  return new Date(s);
};

export function calculateDays(quantity: number, unit: TimeUnit, time = new Date()): number {
  switch (unit) {
    case TimeUnit.Day:
      return quantity;
    case TimeUnit.Week:
      return quantity * 7;
    case TimeUnit.Month:
      return monthsToDays(quantity, time);
    case TimeUnit.Year:
      return monthsToDays(quantity * 12, time);
  }
}

// number of days in a given number of months
// adjusts for days in the current month that have passed
export function monthsToDays(quantity: number, startDate: Date): number {
  // Normalize start date to UTC midnight to avoid DST/timezone artifacts
  const startUTC = new Date(
    Date.UTC(startDate.getUTCFullYear(), startDate.getUTCMonth(), startDate.getUTCDate()),
  );
  const endUTC = addMonths(startUTC, quantity);

  // Calculate the difference in days using UTC-normalized dates
  const duration = endUTC.getTime() - startUTC.getTime();
  return Math.floor(duration / (1000 * 60 * 60 * 24));
}

function addMonths(date: Date, quantity: number): Date {
  // Use UTC components to avoid timezone/DST issues
  const baseYear = date.getUTCFullYear();
  const baseMonth = date.getUTCMonth(); // 0-indexed
  const baseDay = date.getUTCDate();

  // Calculate year and month similar to Rust, but keeping 0-indexed months
  const totalMonths = baseMonth + quantity;
  const year = baseYear + Math.floor(totalMonths / 12);
  const monthZeroIndexed = ((totalMonths % 12) + 12) % 12;

  // Find the max valid day in the target month (UTC)
  const lastOfTargetMonthUTC = new Date(Date.UTC(year, monthZeroIndexed + 1, 0));
  const maxDay = lastOfTargetMonthUTC.getUTCDate();

  const day = Math.min(baseDay, maxDay);

  // Construct UTC date to avoid local timezone shifting the calendar day
  return new Date(Date.UTC(year, monthZeroIndexed, day));
}

function nextStartOfUnit(date: Date, unit: TimeUnit): Date {
  switch (unit) {
    case TimeUnit.Day: {
      // Next day at midnight
      const nextDay = new Date(date);
      nextDay.setUTCDate(nextDay.getUTCDate() + 1);
      nextDay.setUTCHours(0, 0, 0, 0);
      return nextDay;
    }
    case TimeUnit.Week: {
      // Next Monday
      const weekday = date.getUTCDay(); // 0 = Sunday, 1 = Monday, etc.
      const daysToNextMonday = weekday === 1 ? 7 : (8 - weekday) % 7;
      const nextMonday = new Date(date);
      nextMonday.setUTCDate(nextMonday.getUTCDate() + daysToNextMonday);
      nextMonday.setUTCHours(0, 0, 0, 0);
      return nextMonday;
    }
    case TimeUnit.Month: {
      // Next month 1st
      const nextMonth = new Date(date);
      nextMonth.setUTCMonth(nextMonth.getUTCMonth() + 1);
      nextMonth.setUTCDate(1);
      nextMonth.setUTCHours(0, 0, 0, 0);
      return nextMonth;
    }
    case TimeUnit.Year: {
      // Next year January 1st
      const nextYear = new Date(date);
      nextYear.setUTCFullYear(nextYear.getUTCFullYear() + 1);
      nextYear.setUTCMonth(0); // January = 0
      nextYear.setUTCDate(1);
      nextYear.setUTCHours(0, 0, 0, 0);
      return nextYear;
    }
  }
}

function isStartOfUnit(date: Date, unit: TimeUnit): boolean {
  switch (unit) {
    case TimeUnit.Day:
      return date.getUTCHours() === 0 && date.getUTCMinutes() === 0 && date.getUTCSeconds() === 0;
    case TimeUnit.Week: {
      const weekday = date.getUTCDay(); // 0 = Sunday, 1 = Monday, etc.
      return (
        weekday === 1 &&
        date.getUTCHours() === 0 &&
        date.getUTCMinutes() === 0 &&
        date.getUTCSeconds() === 0
      );
    }
    case TimeUnit.Month:
      return (
        date.getUTCDate() === 1 &&
        date.getUTCHours() === 0 &&
        date.getUTCMinutes() === 0 &&
        date.getUTCSeconds() === 0
      );
    case TimeUnit.Year:
      return (
        date.getUTCMonth() === 0 &&
        date.getUTCDate() === 1 &&
        date.getUTCHours() === 0 &&
        date.getUTCMinutes() === 0 &&
        date.getUTCSeconds() === 0
      );
  }
}

function isAlignedToUnit(date: Date, unit: TimeUnit, quantity: number): boolean {
  switch (unit) {
    case TimeUnit.Day:
      return isStartOfUnit(date, TimeUnit.Day);
    case TimeUnit.Week:
      return isStartOfUnit(date, TimeUnit.Week);
    case TimeUnit.Month: {
      if (!isStartOfUnit(date, TimeUnit.Month)) {
        return false;
      }
      const monthZeroIndexed = date.getUTCMonth(); // getUTCMonth() is already 0-indexed
      const q = ((quantity % 12) + 12) % 12 || 1; // Equivalent to quantity.rem_euclid(12).max(1)
      return monthZeroIndexed % q === 0;
    }
    case TimeUnit.Year:
      return isStartOfUnit(date, TimeUnit.Year);
  }
}

function nextAlignedStart(date: Date, unit: TimeUnit, quantity: number): Date {
  switch (unit) {
    case TimeUnit.Day:
      return nextStartOfUnit(date, TimeUnit.Day);
    case TimeUnit.Week:
      return nextStartOfUnit(date, TimeUnit.Week);
    case TimeUnit.Month: {
      const currentMonthZeroIndexed = date.getUTCMonth(); // getUTCMonth() is already 0-indexed
      const q = ((quantity % 12) + 12) % 12 || 1; // Equivalent to quantity.rem_euclid(12).max(1)
      const remainder = currentMonthZeroIndexed % q;
      const monthsToAdd = remainder === 0 ? q : q - remainder;

      const targetFirstOfMonth = new Date(date);
      targetFirstOfMonth.setUTCMonth(targetFirstOfMonth.getUTCMonth() + monthsToAdd);
      targetFirstOfMonth.setUTCDate(1);
      targetFirstOfMonth.setUTCHours(0, 0, 0, 0);
      return targetFirstOfMonth;
    }
    case TimeUnit.Year:
      return nextStartOfUnit(date, TimeUnit.Year);
  }
}

export function timeSegment(
  timeRange: TimeRange,
  quantity: number,
  unit: TimeUnit,
  segmentType: TimeSegmentType,
): TimeRange[] {
  if (quantity <= 0) {
    throw new Error('Quantity must be positive');
  }

  const totalDuration = timeRange.to.getTime() - timeRange.from.getTime();
  const totalDays = Math.floor(totalDuration / (1000 * 60 * 60 * 24));

  if (totalDays < 0) {
    throw new Error('Invalid time range: after date is before before date');
  }

  switch (segmentType) {
    case TimeSegmentType.FromStartOfRange: {
      // Original logic: segment from the start of the range
      const segments: [Date, Date][] = [];
      let currentStart = new Date(timeRange.from);

      while (currentStart < timeRange.to) {
        // Calculate the end of the current segment
        const segmentDays = calculateDays(quantity, unit, currentStart);
        const segmentDuration = segmentDays * 24 * 60 * 60 * 1000;

        const currentEnd = new Date(
          Math.min(currentStart.getTime() + segmentDuration, timeRange.to.getTime()),
        );

        segments.push([currentStart, currentEnd]);
        currentStart = new Date(currentEnd);
      }

      return segments.map(([from, to]) => new TimeRange(from, to));
    }
    case TimeSegmentType.FromStartOfUnit: {
      // Align to the start of units (quantity-aware), then use the requested quantity
      const segments: [Date, Date][] = [];
      let currentStart = new Date(timeRange.from);

      while (currentStart < timeRange.to) {
        if (isAlignedToUnit(currentStart, unit, quantity)) {
          // We are aligned: create a full segment of the requested quantity
          const segmentDays = calculateDays(quantity, unit, currentStart);
          const segmentDuration = segmentDays * 24 * 60 * 60 * 1000;
          const desiredEnd = new Date(currentStart.getTime() + segmentDuration);
          const actualEnd = new Date(Math.min(desiredEnd.getTime(), timeRange.to.getTime()));
          segments.push([currentStart, actualEnd]);
          currentStart = new Date(actualEnd);
        } else {
          // Not aligned: first segment goes up to the next aligned boundary
          const nextUnitStart = nextAlignedStart(currentStart, unit, quantity);
          const actualEnd = new Date(Math.min(nextUnitStart.getTime(), timeRange.to.getTime()));
          segments.push([currentStart, actualEnd]);
          currentStart = new Date(actualEnd);
        }
      }

      return segments.map(([from, to]) => new TimeRange(from, to));
    }
  }
}
