import { startOfDay, calculateDays, monthsToDays, timeSegment } from './index';
import { TimeRange } from './time_range';
import { TimeSegmentType } from './time_segment_type';
import { TimeUnit } from './time_unit';

describe('startOfDay', () => {
  test('should return correct date string', () => {
    const ctx = new Date('2023-05-15T12:00:00Z');
    const result = startOfDay(0, ctx);
    expect(result).toBe('2023-05-15');
  });
});

describe('calculateDays', () => {
  test.each`
    testName                           | currentDate       | quantity | timeUnit          | expected
    ${'days'}                          | ${[2023, 1, 1]}   | ${1}     | ${TimeUnit.Day}   | ${1}
    ${'days negative'}                 | ${[2023, 1, 1]}   | ${-1}    | ${TimeUnit.Day}   | ${-1}
    ${'weeks'}                         | ${[2023, 1, 1]}   | ${1}     | ${TimeUnit.Week}  | ${7}
    ${'weeks negative'}                | ${[2023, 1, 1]}   | ${-2}    | ${TimeUnit.Week}  | ${-14}
    ${'months jan to feb'}             | ${[2023, 1, 1]}   | ${1}     | ${TimeUnit.Month} | ${31}
    ${'months jan to mar'}             | ${[2023, 1, 1]}   | ${2}     | ${TimeUnit.Month} | ${59}
    ${'months full non leap'}          | ${[2023, 1, 1]}   | ${12}    | ${TimeUnit.Month} | ${365}
    ${'years'}                         | ${[2023, 1, 1]}   | ${1}     | ${TimeUnit.Year}  | ${365}
    ${'years negative'}                | ${[2023, 1, 1]}   | ${-1}    | ${TimeUnit.Year}  | ${-365}
    ${'months leap year'}              | ${[2024, 1, 1]}   | ${1}     | ${TimeUnit.Month} | ${31}
    ${'months leap year feb'}          | ${[2024, 1, 1]}   | ${2}     | ${TimeUnit.Month} | ${60}
    ${'months full leap'}              | ${[2024, 1, 1]}   | ${12}    | ${TimeUnit.Month} | ${366}
    ${'years leap'}                    | ${[2024, 1, 1]}   | ${1}     | ${TimeUnit.Year}  | ${366}
    ${'years negative leap'}           | ${[2024, 1, 1]}   | ${-1}    | ${TimeUnit.Year}  | ${-365}
    ${'across year boundary'}          | ${[2023, 12, 15]} | ${2}     | ${TimeUnit.Month} | ${62}
    ${'across year boundary negative'} | ${[2024, 2, 15]}  | ${-2}    | ${TimeUnit.Month} | ${-62}
    ${'multiple years'}                | ${[2023, 12, 15]} | ${4}     | ${TimeUnit.Year}  | ${365 * 3 + 366}
    ${'century'}                       | ${[2023, 12, 15]} | ${100}   | ${TimeUnit.Year}  | ${36524}
  `('$testName', ({ currentDate, quantity, timeUnit, expected }) => {
    const [y, m, d] = currentDate;
    const ctx = new Date(Date.UTC(y, m - 1, d, 0, 0, 0));
    const result = calculateDays(quantity, timeUnit, ctx);
    expect(result).toBe(expected);
  });
});

describe('monthsToDays', () => {
  test.each`
    testName                   | mockDate         | months | expected
    ${'months one'}            | ${[2023, 1, 1]}  | ${1}   | ${31}
    ${'months full year'}      | ${[2023, 1, 1]}  | ${12}  | ${365}
    ${'months leap one'}       | ${[2024, 1, 1]}  | ${1}   | ${31}
    ${'months leap two'}       | ${[2024, 1, 1]}  | ${2}   | ${60}
    ${'months leap full year'} | ${[2024, 1, 1]}  | ${12}  | ${366}
    ${'across leap year'}      | ${[2023, 12, 1]} | ${3}   | ${91}
  `('$testName', ({ mockDate, months, expected }) => {
    const [y, m, d] = mockDate;
    const ctx = new Date(Date.UTC(y, m - 1, d, 0, 0, 0));
    const result = monthsToDays(months, ctx);
    expect(result).toBe(expected);
  });
});

describe('timeSegment', () => {
  describe('FromStartOfRange', () => {
    test.each`
      testName                               | from                  | to                    | quantity | unit              | expected
      ${'days'}                              | ${'2023-01-01'}       | ${'2023-01-05'}       | ${2}     | ${TimeUnit.Day}   | ${[['2023-01-01', '2023-01-03'], ['2023-01-03', '2023-01-05']]}
      ${'weeks'}                             | ${'2023-01-01'}       | ${'2023-01-15'}       | ${1}     | ${TimeUnit.Week}  | ${[['2023-01-01', '2023-01-08'], ['2023-01-08', '2023-01-15']]}
      ${'months'}                            | ${'2023-01-01'}       | ${'2023-03-01'}       | ${1}     | ${TimeUnit.Month} | ${[['2023-01-01', '2023-02-01'], ['2023-02-01', '2023-03-01']]}
      ${'years'}                             | ${'2023-01-01'}       | ${'2025-01-01'}       | ${1}     | ${TimeUnit.Year}  | ${[['2023-01-01', '2024-01-01'], ['2024-01-01', '2025-01-01']]}
      ${'months leap year'}                  | ${'2024-01-01'}       | ${'2024-03-01'}       | ${1}     | ${TimeUnit.Month} | ${[['2024-01-01', '2024-02-01'], ['2024-02-01', '2024-03-01']]}
      ${'exact fit'}                         | ${'2023-01-01'}       | ${'2023-01-03'}       | ${1}     | ${TimeUnit.Day}   | ${[['2023-01-01', '2023-01-02'], ['2023-01-02', '2023-01-03']]}
      ${'smaller than segment'}              | ${'2023-01-01'}       | ${'2023-01-02T12:00'} | ${2}     | ${TimeUnit.Day}   | ${[['2023-01-01', '2023-01-02T12:00']]}
      ${'with time components'}              | ${'2023-01-01T10:30'} | ${'2023-01-03T15:20'} | ${1}     | ${TimeUnit.Day}   | ${[['2023-01-01T10:30', '2023-01-02T10:30'], ['2023-01-02T10:30', '2023-01-03T10:30'], ['2023-01-03T10:30', '2023-01-03T15:20']]}
      ${'months arbitrary days'}             | ${'2025-01-20'}       | ${'2025-04-15'}       | ${1}     | ${TimeUnit.Month} | ${[['2025-01-20', '2025-02-20'], ['2025-02-20', '2025-03-20'], ['2025-03-20', '2025-04-15']]}
      ${'months arbitrary days short range'} | ${'2025-01-15'}       | ${'2025-02-10'}       | ${1}     | ${TimeUnit.Month} | ${[['2025-01-15', '2025-02-10']]}
      ${'months arbitrary days leap year'}   | ${'2024-01-15'}       | ${'2024-03-15'}       | ${1}     | ${TimeUnit.Month} | ${[['2024-01-15', '2024-02-15'], ['2024-02-15', '2024-03-15']]}
    `('$testName', ({ from, to, quantity, unit, expected }) => {
      const timeRange = TimeRange.fromString(from, to);
      const segments = timeSegment(timeRange, quantity, unit, TimeSegmentType.FromStartOfRange);

      const expectedSegments = expected.map(([fromStr, toStr]: [string, string]) =>
        TimeRange.fromString(fromStr, toStr),
      );

      expect(segments).toEqual(expectedSegments);
    });
  });

  describe('FromStartOfUnit', () => {
    test.each`
      testName             | from                  | to                    | quantity | unit              | expected
      ${'months'}          | ${'2025-01-20'}       | ${'2025-03-15'}       | ${1}     | ${TimeUnit.Month} | ${[['2025-01-20', '2025-02-01'], ['2025-02-01', '2025-03-01'], ['2025-03-01', '2025-03-15']]}
      ${'months (case 2)'} | ${'2025-01-01'}       | ${'2025-04-01'}       | ${1}     | ${TimeUnit.Month} | ${[['2025-01-01', '2025-02-01'], ['2025-02-01', '2025-03-01'], ['2025-03-01', '2025-04-01']]}
      ${'quarterly'}       | ${'2025-01-15'}       | ${'2025-12-31'}       | ${3}     | ${TimeUnit.Month} | ${[['2025-01-15', '2025-04-01'], ['2025-04-01', '2025-07-01'], ['2025-07-01', '2025-10-01'], ['2025-10-01', '2025-12-31']]}
      ${'days'}            | ${'2025-01-20T10:30'} | ${'2025-01-22T15:45'} | ${1}     | ${TimeUnit.Day}   | ${[['2025-01-20T10:30', '2025-01-21T00:00'], ['2025-01-21T00:00', '2025-01-22T00:00'], ['2025-01-22T00:00', '2025-01-22T15:45']]}
      ${'days (case 2)'}   | ${'2025-01-01'}       | ${'2025-01-03'}       | ${1}     | ${TimeUnit.Day}   | ${[['2025-01-01', '2025-01-02'], ['2025-01-02', '2025-01-03']]}
      ${'days (case 3)'}   | ${'2025-01-01'}       | ${'2025-01-06'}       | ${2}     | ${TimeUnit.Day}   | ${[['2025-01-01', '2025-01-03'], ['2025-01-03', '2025-01-05'], ['2025-01-05', '2025-01-06']]}
      ${'weeks'}           | ${'2025-01-18'}       | ${'2025-02-03'}       | ${1}     | ${TimeUnit.Week}  | ${[['2025-01-18', '2025-01-20'], ['2025-01-20', '2025-01-27'], ['2025-01-27', '2025-02-03']]}
      ${'weeks (case 2)'}  | ${'2025-01-20'}       | ${'2025-02-03'}       | ${1}     | ${TimeUnit.Week}  | ${[['2025-01-20', '2025-01-27'], ['2025-01-27', '2025-02-03']]}
      ${'weeks (case 3)'}  | ${'2025-01-20'}       | ${'2025-02-18'}       | ${2}     | ${TimeUnit.Week}  | ${[['2025-01-20', '2025-02-03'], ['2025-02-03', '2025-02-17'], ['2025-02-17', '2025-02-18']]}
      ${'years'}           | ${'2025-06-15'}       | ${'2027-03-10'}       | ${1}     | ${TimeUnit.Year}  | ${[['2025-06-15', '2026-01-01'], ['2026-01-01', '2027-01-01'], ['2027-01-01', '2027-03-10']]}
      ${'years (case 2)'}  | ${'2025-01-01'}       | ${'2027-01-01'}       | ${1}     | ${TimeUnit.Year}  | ${[['2025-01-01', '2026-01-01'], ['2026-01-01', '2027-01-01']]}
    `('$testName', ({ from, to, quantity, unit, expected }) => {
      const timeRange = TimeRange.fromString(from, to);
      const segments = timeSegment(timeRange, quantity, unit, TimeSegmentType.FromStartOfUnit);

      const expectedSegments = expected.map(([fromStr, toStr]: [string, string]) =>
        TimeRange.fromString(fromStr, toStr),
      );

      expect(segments).toEqual(expectedSegments);
    });
  });

  test('should throw error for invalid quantity', () => {
    const timeRange = TimeRange.fromString('2023-01-01', '2023-01-05');

    expect(() => {
      timeSegment(timeRange, 0, TimeUnit.Day, TimeSegmentType.FromStartOfRange);
    }).toThrow('Quantity must be positive');

    expect(() => {
      timeSegment(timeRange, -1, TimeUnit.Day, TimeSegmentType.FromStartOfRange);
    }).toThrow('Quantity must be positive');
  });

  test('should throw error for invalid range', () => {
    const timeRange = TimeRange.fromString('2023-01-05', '2023-01-01');

    expect(() => {
      timeSegment(timeRange, 1, TimeUnit.Day, TimeSegmentType.FromStartOfRange);
    }).toThrow('Invalid time range: after date is before before date');
  });
});
