// Helper function to capitalize first letter
function upperFirst(str: string): string {
  return str.length > 0 ? str[0]!.toUpperCase() + str.slice(1) : str;
}

let counters: Record<string, number> = {};

export function resetUniqueIdCounters() {
  counters = {};
}

export function uniqueId(key: string) {
  if (!counters[key]) counters[key] = 0;
  counters[key]++;

  return `${key}${counters[key]}`;
}

export function wildcardMatch(str: string, pattern: string) {
  const lowerPattern = pattern.toLowerCase();
  const lowerStr = str.toLowerCase();

  if (!lowerPattern.includes('*')) return lowerPattern === lowerStr;

  // Escape regex special characters first (matching Rust implementation)
  const escapedPattern = lowerPattern
    .split('')
    .map((c) => {
      if (['\\', '.', '+', '?', '^', '$', '{', '}', '(', ')', '|', '[', ']'].includes(c)) {
        return `\\${c}`;
      }
      return c;
    })
    .join('');

  // Then replace literal asterisks with .*
  const regexPattern = `^${escapedPattern.replace(/\*/g, '.*')}$`;

  try {
    return new RegExp(regexPattern).test(lowerStr);
  } catch {
    return false;
  }
}

export function toSentenceCase(str: string) {
  if (str.toLowerCase() === 'id' || str.toLowerCase() === 'iid' || str.length <= 1) {
    return str.toUpperCase();
  }

  // Custom implementation to mimic Rust's heck::ToTitleCase behavior
  // Split on word boundaries (underscores, hyphens, camelCase, etc.)
  const words = str
    .replace(/([a-z])([A-Z])/g, '$1 $2') // camelCase -> camel Case
    .replace(/[@._-]/g, ' ') // Replace @, dots, underscores, and hyphens with spaces
    .replace(/\*/g, '') // Remove asterisks completely (matching Rust behavior)
    .split(' ')
    .filter((word) => word.length > 0);

  if (words.length === 0) return str;

  // Only capitalize the first word, lowercase the rest
  const result = words
    .map((word, index) => (index === 0 ? upperFirst(word.toLowerCase()) : word.toLowerCase()))
    .join(' ');

  return result;
}

type EpicReference = { groupPath: string; id: string };
type WorkItemReference = { groupPath?: string; projectPath?: string; id: string };

/// Parses an epic reference string and returns (group_path, id)
///
/// Epic references can be in the format:
/// - "123" -> uses fallback_group_path
/// - "&456" -> uses fallback_group_path
/// - "group/path&789" -> uses "group/path"
///
/// This will be used for parsing epic references with "&" prefix.
export function parseEpicReference(reference: string, fallbackGroupPath = ''): EpicReference {
  const cleaned = reference.replace(/^&/, '');
  if (!cleaned.includes('&')) return { groupPath: fallbackGroupPath, id: cleaned };

  const parts = cleaned.split('&');
  if (parts.length > 1) return { groupPath: parts[0] || fallbackGroupPath, id: parts[1]! };

  return { groupPath: fallbackGroupPath, id: cleaned };
}

export function parseWorkItemReference(
  reference: string,
  fallbackGroupPath = '',
): WorkItemReference {
  if (reference.includes('&')) return parseEpicReference(reference, fallbackGroupPath);

  const cleaned = reference.replace(/^#/, '');
  if (!cleaned.includes('#')) return { projectPath: '', id: cleaned };

  const parts = cleaned.split(/#/);
  if (parts.length > 1) return { projectPath: parts[0] || '', id: parts[1]! };

  return { projectPath: '', id: cleaned };
}
