import {
  resetUniqueIdCounters,
  uniqueId,
  toSentenceCase,
  parseEpicReference,
  parseWorkItemReference,
  wildcardMatch,
} from './common';

describe('common', () => {
  beforeEach(() => {
    resetUniqueIdCounters();
  });

  describe('uniqueId', () => {
    it('should generate unique IDs', () => {
      const base = 'test';
      expect(uniqueId(base)).toBe('test1');
      expect(uniqueId(base)).toBe('test2');
      expect(uniqueId(base)).toBe('test3');
    });
  });

  describe('toSentenceCase', () => {
    it('should handle empty string', () => {
      expect(toSentenceCase('')).toBe('');
    });

    it('should handle single character', () => {
      expect(toSentenceCase('a')).toBe('A');
      expect(toSentenceCase('A')).toBe('A');
    });

    it('should handle special cases for "id" and "iid"', () => {
      expect(toSentenceCase('id')).toBe('ID');
      expect(toSentenceCase('iid')).toBe('IID');
      expect(toSentenceCase('Id')).toBe('ID');
      expect(toSentenceCase('IID')).toBe('IID');
    });

    it('should handle simple words', () => {
      expect(toSentenceCase('hello')).toBe('Hello');
      expect(toSentenceCase('world')).toBe('World');
      expect(toSentenceCase('HELLO')).toBe('Hello');
      expect(toSentenceCase('WORLD')).toBe('World');
    });

    it('should handle multiple words', () => {
      expect(toSentenceCase('hello world')).toBe('Hello world');
      expect(toSentenceCase('HELLO WORLD')).toBe('Hello world');
      expect(toSentenceCase('hello_world')).toBe('Hello world');
      expect(toSentenceCase('hello-world')).toBe('Hello world');
    });

    it('should handle camelCase and PascalCase', () => {
      expect(toSentenceCase('camelCase')).toBe('Camel case');
      expect(toSentenceCase('PascalCase')).toBe('Pascal case');
      expect(toSentenceCase('snake_case')).toBe('Snake case');
      expect(toSentenceCase('kebab-case')).toBe('Kebab case');
    });

    it('should handle numbers', () => {
      expect(toSentenceCase('user123')).toBe('User123');
      expect(toSentenceCase('123user')).toBe('123user');
      expect(toSentenceCase('user_123_name')).toBe('User 123 name');
    });

    it('should handle special characters', () => {
      expect(toSentenceCase('user@example.com')).toBe('User example com');
      expect(toSentenceCase('file-name.txt')).toBe('File name txt');
    });
  });

  describe('parseEpicReference', () => {
    const fallbackGroup = 'default-group';

    it('should handle simple ID without group path', () => {
      const result = parseEpicReference('123', fallbackGroup);
      expect(result.groupPath).toBe('default-group');
      expect(result.id).toBe('123');
    });

    it('should handle ID with & prefix but no group path', () => {
      const result = parseEpicReference('&456', fallbackGroup);
      expect(result.groupPath).toBe('default-group');
      expect(result.id).toBe('456');
    });

    it('should handle group path with ID', () => {
      const result = parseEpicReference('gitlab-org&789', fallbackGroup);
      expect(result.groupPath).toBe('gitlab-org');
      expect(result.id).toBe('789');
    });

    it('should handle empty group path (should use fallback)', () => {
      const result = parseEpicReference('&101', fallbackGroup);
      expect(result.groupPath).toBe('default-group');
      expect(result.id).toBe('101');
    });

    it('should handle complex group path', () => {
      const result = parseEpicReference('gitlab-org/gitlab&202', fallbackGroup);
      expect(result.groupPath).toBe('gitlab-org/gitlab');
      expect(result.id).toBe('202');
    });

    it('should handle edge case: empty string', () => {
      const result = parseEpicReference('', fallbackGroup);
      expect(result.groupPath).toBe('default-group');
      expect(result.id).toBe('');
    });
  });

  describe('parseWorkItemReference', () => {
    const fallbackGroup = 'default-group';

    describe('epic references (with &)', () => {
      it('should handle simple ID without group path', () => {
        const result = parseWorkItemReference('123', fallbackGroup);
        expect(result.projectPath).toBe('');
        expect(result.groupPath).toBeUndefined();
        expect(result.id).toBe('123');
      });

      it('should handle ID with & prefix but no group path', () => {
        const result = parseWorkItemReference('&456', fallbackGroup);
        expect(result.groupPath).toBe('default-group');
        expect(result.projectPath).toBeUndefined();
        expect(result.id).toBe('456');
      });

      it('should handle group path with ID', () => {
        const result = parseWorkItemReference('gitlab-org&789', fallbackGroup);
        expect(result.groupPath).toBe('gitlab-org');
        expect(result.projectPath).toBeUndefined();
        expect(result.id).toBe('789');
      });

      it('should handle complex group path', () => {
        const result = parseWorkItemReference('gitlab-org/gitlab&202', fallbackGroup);
        expect(result.groupPath).toBe('gitlab-org/gitlab');
        expect(result.projectPath).toBeUndefined();
        expect(result.id).toBe('202');
      });
    });

    describe('project references (with #)', () => {
      it('should handle simple ID without project path', () => {
        const result = parseWorkItemReference('123', fallbackGroup);
        expect(result.projectPath).toBe('');
        expect(result.groupPath).toBeUndefined();
        expect(result.id).toBe('123');
      });

      it('should handle ID with # prefix but no project path', () => {
        const result = parseWorkItemReference('#456', fallbackGroup);
        expect(result.projectPath).toBe('');
        expect(result.groupPath).toBeUndefined();
        expect(result.id).toBe('456');
      });

      it('should handle project path with ID', () => {
        const result = parseWorkItemReference('gitlab-org/gitlab#789', fallbackGroup);
        expect(result.projectPath).toBe('gitlab-org/gitlab');
        expect(result.groupPath).toBeUndefined();
        expect(result.id).toBe('789');
      });

      it('should handle complex project path', () => {
        const result = parseWorkItemReference('gitlab-org/gitlab-ce#202', fallbackGroup);
        expect(result.projectPath).toBe('gitlab-org/gitlab-ce');
        expect(result.groupPath).toBeUndefined();
        expect(result.id).toBe('202');
      });

      it('should handle empty project path (should use empty string)', () => {
        const result = parseWorkItemReference('#101', fallbackGroup);
        expect(result.projectPath).toBe('');
        expect(result.groupPath).toBeUndefined();
        expect(result.id).toBe('101');
      });
    });

    describe('edge cases', () => {
      it('should handle empty string', () => {
        const result = parseWorkItemReference('', fallbackGroup);
        expect(result.projectPath).toBe('');
        expect(result.groupPath).toBeUndefined();
        expect(result.id).toBe('');
      });

      it('should handle string with only #', () => {
        const result = parseWorkItemReference('#', fallbackGroup);
        expect(result.projectPath).toBe('');
        expect(result.groupPath).toBeUndefined();
        expect(result.id).toBe('');
      });

      it('should handle string with only &', () => {
        const result = parseWorkItemReference('&', fallbackGroup);
        expect(result.groupPath).toBe('default-group');
        expect(result.projectPath).toBeUndefined();
        expect(result.id).toBe('');
      });

      it('should handle multiple # separators (takes first one)', () => {
        const result = parseWorkItemReference('project#123#456', fallbackGroup);
        expect(result.projectPath).toBe('project');
        expect(result.groupPath).toBeUndefined();
        expect(result.id).toBe('123');
      });

      it('should handle multiple & separators (takes first one)', () => {
        const result = parseWorkItemReference('group&123&456', fallbackGroup);
        expect(result.groupPath).toBe('group');
        expect(result.projectPath).toBeUndefined();
        expect(result.id).toBe('123');
      });
    });

    describe('without fallback group path', () => {
      it('should handle epic reference without fallback', () => {
        const result = parseWorkItemReference('&123');
        expect(result.groupPath).toBe('');
        expect(result.projectPath).toBeUndefined();
        expect(result.id).toBe('123');
      });

      it('should handle project reference without fallback', () => {
        const result = parseWorkItemReference('#123');
        expect(result.projectPath).toBe('');
        expect(result.groupPath).toBeUndefined();
        expect(result.id).toBe('123');
      });
    });
  });

  describe('wildcardMatch', () => {
    it('should handle exact matches', () => {
      expect(wildcardMatch('label', 'label')).toBe(true);
      expect(wildcardMatch('label', 'a-label')).toBe(false);
      expect(wildcardMatch('label', 'label-a')).toBe(false);
    });

    it('should handle prefix wildcard (*label)', () => {
      expect(wildcardMatch('a-label', '*label')).toBe(true);
      expect(wildcardMatch('a-label-a', '*label')).toBe(false);
    });

    it('should handle suffix wildcard (label*)', () => {
      expect(wildcardMatch('label-a', 'label*')).toBe(true);
      expect(wildcardMatch('a-label-a', 'label*')).toBe(false);
    });

    it('should handle both prefix and suffix wildcard (*label*)', () => {
      expect(wildcardMatch('a-label-a', '*label*')).toBe(true);
    });

    it('should handle middle wildcard (l*l)', () => {
      expect(wildcardMatch('label', 'l*l')).toBe(true);
    });

    it('should handle special characters', () => {
      expect(wildcardMatch('!@#$%^&*()-=+/?[]{}', '!@#$%^&*()-=+/?[]{}')).toBe(true);
    });
  });
});
