import { ReferenceTokenType, ReferenceType } from './reference_type';
import { TimeUnit } from '../utils/date_time/time_unit';
import { Variable } from './variable';
import { CompileContext } from './compile_context';

export enum ValueType {
  Null = 'Null',
  Bool = 'Bool',
  Number = 'Number',
  Quoted = 'Quoted',
  RelativeDate = 'RelativeDate',
  Date = 'Date',
  List = 'List',
  Function = 'Function',
  Token = 'Token',
  Reference = 'Reference',
  Subquery = 'Subquery',
}

export abstract class Value<T extends ValueType = ValueType, V = unknown> {
  constructor(
    public type: T,
    public value: V,
  ) {}

  abstract clone(): Value<T, V>;

  eq(other: Value) {
    return this.type === other.type && this.value === other.value;
  }
}

export namespace Value {
  export class Null extends Value<ValueType.Null, null> {
    constructor() {
      super(ValueType.Null, null);
    }

    override clone() {
      return new Null();
    }

    override toString() {
      return 'null';
    }
  }

  export class Bool extends Value<ValueType.Bool, boolean> {
    constructor(value: boolean) {
      super(ValueType.Bool, value);
    }

    override clone() {
      return new Bool(this.value);
    }

    override toString() {
      return this.value.toString();
    }
  }

  export class Number extends Value<ValueType.Number, number> {
    constructor(value: number) {
      super(ValueType.Number, value);
    }

    override clone() {
      return new Number(this.value);
    }

    override toString() {
      return this.value.toString();
    }
  }

  export class Quoted extends Value<ValueType.Quoted, string> {
    constructor(value: string) {
      super(ValueType.Quoted, value);
    }

    override clone() {
      return new Quoted(this.value);
    }

    override toString() {
      return `"${this.value}"`;
    }
  }

  export class RelativeDate extends Value<ValueType.RelativeDate, number> {
    constructor(
      value: number,
      public unit: TimeUnit,
    ) {
      super(ValueType.RelativeDate, value);
    }

    override clone() {
      return new RelativeDate(this.value, this.unit);
    }

    override toString() {
      return `${this.value}${this.unit}`;
    }

    override eq(other: Value) {
      if (super.eq(other) && other instanceof RelativeDate) return this.unit === other.unit;

      return false;
    }
  }

  export class Date extends Value<ValueType.Date, string> {
    constructor(value: string) {
      super(ValueType.Date, value);
    }

    override clone() {
      return new Date(this.value);
    }

    override toString() {
      return `"${this.value}"`;
    }
  }

  export class List extends Value<ValueType.List, Value[]> {
    constructor(value: Value[]) {
      super(ValueType.List, value);
    }

    override clone() {
      return new List(this.value.map((v) => v.clone()));
    }

    override toString() {
      return `(${this.value.map((v) => v.toString()).join(', ')})`;
    }
  }

  export class Function extends Value<ValueType.Function, string> {
    constructor(
      value: string,
      public args: string[],
    ) {
      super(ValueType.Function, value);
    }

    override clone() {
      return new Function(this.value, this.args);
    }

    override toString() {
      return `${this.value}(${this.args.map((a) => `"${a}"`).join(', ')})`;
    }

    override eq(other: Value) {
      if (super.eq(other) && other instanceof Function) {
        return this.args.every((arg, i) => other.args[i] === arg);
      }

      return false;
    }
  }

  export class Token extends Value<ValueType.Token, string> {
    constructor(value: string) {
      super(ValueType.Token, value);
    }

    override clone() {
      return new Token(this.value);
    }

    override toString() {
      return this.value;
    }

    override eq(other: Value) {
      return other instanceof Token && this.value.toUpperCase() === other.value.toUpperCase();
    }
  }

  export class Reference<T extends ReferenceType = ReferenceType> extends Value<
    ValueType.Reference,
    string
  > {
    constructor(
      public referenceType: T,
      value: string,
    ) {
      super(ValueType.Reference, value);
    }

    override clone() {
      return new Reference(this.referenceType, this.value);
    }

    override toString() {
      return `${this.referenceType.type}${this.value}`;
    }

    override eq(other: Value) {
      if (super.eq(other) && other instanceof Reference) {
        return this.referenceType === other.referenceType;
      }

      return false;
    }
  }

  export namespace Reference {
    export type User = Reference<ReferenceType<ReferenceTokenType.User>>;
    export type Milestone = Reference<ReferenceType<ReferenceTokenType.Milestone>>;
    export type Label = Reference<ReferenceType<ReferenceTokenType.Label>>;
    export type Iteration = Reference<ReferenceType<ReferenceTokenType.Iteration>>;
    export type Epic = Reference<ReferenceType<ReferenceTokenType.Epic>>;
    export type WorkItem = Reference<ReferenceType<ReferenceTokenType.WorkItem>>;
  }

  export class Subquery extends Value<ValueType.Subquery, string> {
    constructor(
      value: string,
      public variable: Variable,
      public context: CompileContext,
    ) {
      super(ValueType.Subquery, value);
    }

    override clone() {
      return new Subquery(this.value, this.variable, this.context);
    }

    override toString() {
      return `($${this.variable.key})`;
    }
  }
}
