import { SourceAnalyzer } from '../analyzer/sources';
import { EpicsSourceAnalyzer } from '../analyzer/sources/epics';
import { MergeRequestsSourceAnalyzer } from '../analyzer/sources/merge_requests';
import { IssuesSourceAnalyzer } from '../analyzer/sources/issues';
import { featureFlags } from '../utils/feature_flags';

export const enum SourceName {
  Epics = 'epic',
  Issues = 'issue',
  MergeRequests = 'mergeRequest',
}

export class Source {
  static fromString(s: string) {
    switch (s.toLowerCase()) {
      case 'epic':
        return new Source(featureFlags.glqlWorkItems ? SourceName.Issues : SourceName.Epics);
      case 'mergerequest':
        return new Source(SourceName.MergeRequests);
      default:
        return new Source(SourceName.Issues);
    }
  }

  constructor(public name: SourceName) {}

  clone() {
    return new Source(this.name);
  }

  get analyzer(): SourceAnalyzer {
    switch (this.name) {
      case SourceName.Epics:
        return new EpicsSourceAnalyzer();
      case SourceName.MergeRequests:
        return new MergeRequestsSourceAnalyzer();
      case SourceName.Issues:
      default:
        return new IssuesSourceAnalyzer();
    }
  }

  toString() {
    switch (this.name) {
      case SourceName.Epics:
        return 'epics';
      case SourceName.Issues:
        return 'work items';
      case SourceName.MergeRequests:
        return 'merge requests';
    }
  }
}
