import { Field } from './field';
import { Expression } from './expression';
import { Value } from './value';
import { Context } from './context';
import { DisplayField } from './display_field';
import { Dimension } from './dimension';
import { Sort } from './sort';
import { applyFunctions } from '../transformer/functions';
import { extractContext, removeExtractedFields } from '../transformer/context';
import { analyze } from '../analyzer';
import { generateCode } from '../codegen/graphql';
import { transform } from '../transformer';
import { SegmentedQuery } from './segmented_query';
import { Operator } from './operator';
import { DimensionSegment } from './dimension_segment';

export class Query {
  expressions: Expression[] = [];
  context: Context = new Context();
  fields: DisplayField[] = [];
  groupBy: Dimension[] = [];
  sort?: Sort;
  aggregate: DisplayField[] = [];
  isSubquery: boolean = false;

  constructor(options: Partial<Query>) {
    Object.assign(this, options);
  }

  static compile(options: Partial<Query>): Query {
    const query = new Query(options);

    applyFunctions(query);
    extractContext(query);
    analyze(query);
    removeExtractedFields(query);
    transform(query);

    return query;
  }

  clone() {
    return new Query({
      expressions: this.expressions.map((expr) => expr.clone()),
      context: this.context.clone(),
      fields: this.fields.map((field) => field.clone()),
      groupBy: this.groupBy.map((groupBy) => groupBy.clone()),
      aggregate: this.aggregate.map((aggregate) => aggregate.clone()),
      sort: this.sort?.clone(),
    });
  }

  containsField(field: Field) {
    return this.expressions.some((expr) => expr.field.dealias().eq(field));
  }

  containsFieldWithValue(field: Field, value: Value) {
    return this.expressions.some((expr) => expr.field.dealias().eq(field) && expr.value.eq(value));
  }

  getValue(field: Field, ...operators: Operator[]) {
    return this.expressions.find(
      (expr) =>
        expr.field.dealias().eq(field) && operators.some((operator) => operator.eq(expr.operator)),
    )?.value;
  }

  unsetValue(field: Field, operator: Operator) {
    this.expressions = this.expressions.filter(
      (expr) => !expr.field.dealias().eq(field) || !expr.operator.eq(operator),
    );
    return this;
  }

  setValue(field: Field, operator: Operator, value: Value) {
    this.unsetValue(field, operator);
    this.expressions.push(new Expression(field, operator, value));
    return this;
  }

  toGraphQL() {
    return generateCode(this);
  }

  segment(): SegmentedQuery[] | void {
    const dimension0 = this.groupBy[0];
    if (!dimension0) return;

    const dimension1 = this.groupBy[1];
    const segments = DimensionSegment.combine(dimension0.segment(this), dimension1?.segment(this));

    return segments.map((segments) => {
      const segmentedQuery = this.clone();
      segments[0].apply(segmentedQuery);
      segments[1]?.apply(segmentedQuery);

      return new SegmentedQuery(segmentedQuery, segments);
    });
  }
}
