import { Field } from './field';
import { Operator } from './operator';
import { ReferenceType } from './reference_type';
import { RelationshipType } from './relationship_type';
import { FieldTypeAnalyzer } from '../analyzer/field_type';

export class FieldPairing {
  constructor(
    public field: Field,
    public fieldType: FieldType,
  ) {}
}

export abstract class FieldType {
  constructor() {}

  get analyzer() {
    return new FieldTypeAnalyzer(this);
  }

  get isBooleanLike() {
    return this.any((fieldType) => fieldType instanceof FieldType.BooleanLike);
  }

  get isDateLike() {
    return this.any((fieldType) => fieldType instanceof FieldType.DateLike);
  }

  get isNullable() {
    return this.any((fieldType) => fieldType instanceof FieldType.Nullable);
  }

  get isListLike() {
    return this.any((fieldType) => fieldType instanceof FieldType.ListLike);
  }

  get isHasManyListLike() {
    return this.any(
      (fieldType) =>
        fieldType instanceof FieldType.ListLike &&
        fieldType.relationshipType === RelationshipType.HasMany,
    );
  }

  get isHasOneListLike() {
    return this.any(
      (fieldType) =>
        fieldType instanceof FieldType.ListLike &&
        fieldType.relationshipType === RelationshipType.HasOne,
    );
  }

  get isEnumLike() {
    return this.any(
      (fieldType) =>
        fieldType instanceof FieldType.EnumLike || fieldType instanceof FieldType.StringEnumLike,
    );
  }

  get enumValues(): string[] {
    if (this instanceof FieldType.EnumLike || this instanceof FieldType.StringEnumLike) {
      return this.values;
    }
    return [];
  }

  find(predicate: (fieldType: FieldType) => boolean): FieldType | void {
    if (predicate(this)) return this;
    if (this instanceof FieldType.PairedWith) return this.fieldType.find(predicate);
    if (this instanceof FieldType.WithOperators) return this.fieldType.find(predicate);
    if (this instanceof FieldType.Multiple)
      return this.fieldTypes.find((fieldType) => fieldType.find(predicate));
  }

  any(predicate: (fieldType: FieldType) => boolean): boolean {
    const types: FieldType[] = [this];

    while (types.length > 0) {
      const currentType = types.pop()!;

      if (currentType instanceof FieldType.WithOperators) types.push(currentType.fieldType);
      else if (currentType instanceof FieldType.Multiple) types.push(...currentType.fieldTypes);
      else if (predicate(currentType)) return true;
    }
    return false;
  }

  withOps(...operators: [Operator, ...Operator[]]) {
    return new FieldType.WithOperators(this, operators);
  }

  pairedWith(...fields: [Field, ...Field[]]) {
    return new FieldType.PairedWith(this, fields);
  }

  or(other: FieldType) {
    if (this instanceof FieldType.Multiple && other instanceof FieldType.Multiple) {
      return new FieldType.Multiple([...this.fieldTypes, ...other.fieldTypes]);
    }
    if (this instanceof FieldType.Multiple) {
      return new FieldType.Multiple([...this.fieldTypes, other]);
    }
    if (other instanceof FieldType.Multiple) {
      return new FieldType.Multiple([this, ...other.fieldTypes]);
    }
    return new FieldType.Multiple([this, other]);
  }
}

export namespace FieldType {
  export class Nullable extends FieldType {
    override toString() {
      return '`Nullable` (`null`, `none`, `any`)';
    }
  }

  export class StringLike extends FieldType {
    override toString() {
      return '`String`';
    }
  }

  export class NumberLike extends FieldType {
    override toString() {
      return '`Number`';
    }
  }

  export class DateLike extends FieldType {
    override toString() {
      return '`Date`';
    }
  }

  export class BooleanLike extends FieldType {
    override toString() {
      return '`Boolean` (`true`, `false`)';
    }
  }

  export class ListLike extends FieldType {
    constructor(
      public relationshipType: RelationshipType,
      public fieldType: FieldType,
    ) {
      super();
    }

    override toString() {
      return '`List`';
    }
  }

  export class EnumLike extends FieldType {
    constructor(public values: string[]) {
      super();
    }

    override toString() {
      return `\`Enum\` (${this.values.map((v) => `\`${v}\``).join(', ')})`;
    }
  }

  export class StringEnumLike extends FieldType {
    constructor(public values: string[]) {
      super();
    }

    override toString() {
      return `\`StringEnum\` (${this.values.map((v) => `\`"${v}"\``).join(', ')})`;
    }
  }

  export class ReferenceLike extends FieldType {
    constructor(public referenceType: ReferenceType) {
      super();
    }

    override toString() {
      return this.referenceType.toString();
    }
  }

  export class PairedWith extends FieldType {
    constructor(
      public fieldType: FieldType,
      public fields: Field[],
    ) {
      super();
    }

    override toString() {
      return this.fieldType.toString();
    }
  }

  export class WithOperators extends FieldType {
    constructor(
      public fieldType: FieldType,
      public operators: Operator[],
    ) {
      super();
    }

    override toString() {
      return this.fieldType.toString();
    }
  }

  export class Multiple extends FieldType {
    constructor(public fieldTypes: FieldType[]) {
      super();
    }

    override toString() {
      return this.fieldTypes.join(',');
    }
  }

  export class UnknownFieldType extends FieldType {
    override toString() {
      return '`Unsupported`';
    }
  }
}
