import { featureFlags } from '../utils/feature_flags';

export const enum FieldName {
  Type = 'type',
  Id = 'id',
  Iid = 'iid',
  Reference = 'reference',
  WebUrl = 'webUrl',
  WebPath = 'webPath',

  Label = 'label',
  Epic = 'epic',
  Parent = 'parent',
  Created = 'created',
  Updated = 'updated',
  Assignee = 'assignee',
  Author = 'author',
  Milestone = 'milestone',
  Weight = 'weight',
  Group = 'group',
  Project = 'project',
  Confidential = 'confidential',
  Health = 'health',
  Closed = 'closed',
  Iteration = 'iteration',
  Cadence = 'cadence',
  Start = 'start',
  Due = 'due',
  IncludeSubgroups = 'includeSubgroups',
  Title = 'title',
  Description = 'description',
  Popularity = 'popularity',
  Status = 'status',
  MyReaction = 'myReaction',
  Subscribed = 'subscribed',
  TaskCompletionStatus = 'taskCompletionStatus',
  LastComment = 'lastComment',
  TimeEstimate = 'timeEstimate',
  TotalTimeSpent = 'totalTimeSpent',
  Progress = 'progress',
  Color = 'color',

  Draft = 'draft',
  Approver = 'approver',
  Reviewer = 'reviewer',
  Merger = 'merger',
  State = 'state',
  Merged = 'merged',
  Deployed = 'deployed',
  Environment = 'environment',
  SourceBranch = 'sourceBranch',
  TargetBranch = 'targetBranch',
  SourceProject = 'sourceProject',
  TargetProject = 'targetProject',

  // Aggregation
  Count = 'count',
  TotalTimeToMerge = 'totalTimeToMerge',
  MeanTimeToMerge = 'meanTimeToMerge',

  CustomField = 'customField',
  UnknownField = 'unknownField',
  AliasedField = 'aliasedField',
}

export class Field {
  name: FieldName;

  constructor(name: FieldName) {
    this.name = name;
  }

  clone() {
    return new Field(this.name);
  }

  aliasedAs(alias: string) {
    return new Field.Aliased(this, alias);
  }

  dealias() {
    return this instanceof Field.Aliased ? this.field : this;
  }

  get displayFieldName() {
    switch (this.name) {
      case FieldName.Assignee:
        return 'assignees';
      case FieldName.Closed:
        return 'closedAt';
      case FieldName.Created:
        return 'createdAt';
      case FieldName.Due:
        return 'dueDate';
      case FieldName.Health:
        return 'healthStatus';
      case FieldName.Label:
        return 'labels';
      case FieldName.Updated:
        return 'updatedAt';
      case FieldName.Description:
        return 'descriptionHtml';
      case FieldName.Merged:
        return 'mergedAt';
      case FieldName.Reviewer:
        return 'reviewers';
      case FieldName.Merger:
        return 'mergeUser';
      case FieldName.Approver:
        return 'approvedBy';
      case FieldName.Start:
        return 'startDate';
      case FieldName.Type:
        return featureFlags.glqlWorkItems ? 'workItemType' : 'type';
      case FieldName.Epic:
      case FieldName.Parent:
        return featureFlags.glqlWorkItems ? 'parent' : 'epic';
      default:
        return this.name.toString();
    }
  }

  toString() {
    return this.name.toString();
  }

  eq(other: Field) {
    return this.name === other.name;
  }

  static fromString(s: string) {
    const fieldMapping: Record<string, FieldName> = {
      type: FieldName.Type,
      id: FieldName.Id,
      iid: FieldName.Iid,
      reference: FieldName.Reference,
      weburl: FieldName.WebUrl,
      webpath: FieldName.WebPath,
      label: FieldName.Label,
      labels: FieldName.Label,
      epic: FieldName.Epic,
      parent: FieldName.Parent,
      opened: FieldName.Created,
      openedat: FieldName.Created,
      created: FieldName.Created,
      createdat: FieldName.Created,
      updated: FieldName.Updated,
      updatedat: FieldName.Updated,
      assignee: FieldName.Assignee,
      assignees: FieldName.Assignee,
      author: FieldName.Author,
      milestone: FieldName.Milestone,
      weight: FieldName.Weight,
      group: FieldName.Group,
      project: FieldName.Project,
      confidential: FieldName.Confidential,
      health: FieldName.Health,
      healthstatus: FieldName.Health,
      closed: FieldName.Closed,
      closedat: FieldName.Closed,
      iteration: FieldName.Iteration,
      cadence: FieldName.Cadence,
      start: FieldName.Start,
      startdate: FieldName.Start,
      due: FieldName.Due,
      duedate: FieldName.Due,
      title: FieldName.Title,
      description: FieldName.Description,
      descriptionhtml: FieldName.Description,
      popularity: FieldName.Popularity,
      includesubgroups: FieldName.IncludeSubgroups,
      status: FieldName.Status,
      myreaction: FieldName.MyReaction,
      myreactionemoji: FieldName.MyReaction,
      subscribed: FieldName.Subscribed,
      reviewer: FieldName.Reviewer,
      reviewers: FieldName.Reviewer,
      reviewedby: FieldName.Reviewer,
      mergeuser: FieldName.Merger,
      merger: FieldName.Merger,
      mergedby: FieldName.Merger,
      draft: FieldName.Draft,
      approver: FieldName.Approver,
      approvers: FieldName.Approver,
      approvedby: FieldName.Approver,
      state: FieldName.State,
      merged: FieldName.Merged,
      mergedat: FieldName.Merged,
      deployed: FieldName.Deployed,
      deployedat: FieldName.Deployed,
      environment: FieldName.Environment,
      sourcebranch: FieldName.SourceBranch,
      sourcebranches: FieldName.SourceBranch,
      targetbranch: FieldName.TargetBranch,
      targetbranches: FieldName.TargetBranch,
      count: FieldName.Count,
      totaltimetomerge: FieldName.TotalTimeToMerge,
      timeestimate: FieldName.TimeEstimate,
      totaltimespent: FieldName.TotalTimeSpent,
      taskcompletionstatus: FieldName.TaskCompletionStatus,
      lastcomment: FieldName.LastComment,
      progress: FieldName.Progress,
      color: FieldName.Color,
      sourceproject: FieldName.SourceProject,
      targetproject: FieldName.TargetProject,
    };

    const fieldName = fieldMapping[s.toLowerCase()];
    if (fieldName) return new Field(fieldName).aliasedAs(s);
    return new Field.Unknown(s);
  }
}

export namespace Field {
  export class Custom extends Field {
    constructor(public customName: string) {
      super(FieldName.CustomField);
    }

    override clone() {
      return new Field.Custom(this.customName);
    }

    override toString() {
      return `customField("${this.customName}")`;
    }
  }

  export class Unknown extends Field {
    constructor(public customName: string) {
      super(FieldName.UnknownField);
    }

    override clone() {
      return new Field.Unknown(this.customName);
    }

    override toString() {
      return this.customName;
    }

    override get displayFieldName() {
      return this.customName;
    }

    override eq(other: Field) {
      if (other instanceof Field.Unknown) {
        return this.customName === other.customName;
      }
      return false;
    }
  }

  export class Aliased extends Field {
    constructor(
      public field: Field,
      public alias: string,
    ) {
      super(FieldName.AliasedField);
    }

    override clone() {
      return new Field.Aliased(this.field.clone(), this.alias);
    }

    override toString() {
      return this.alias;
    }

    override get displayFieldName() {
      return this.field.displayFieldName;
    }

    override eq(other: Field) {
      if (other instanceof Field.Aliased) {
        return this.field.eq(other.field);
      }
      return false;
    }
  }
}
