import { Field } from './field';
import { toSentenceCase } from '../utils/common';

export abstract class DisplayField {
  // Used for indexing the data in the API response
  abstract get key(): string;

  // Used for fetching data from the API
  abstract get name(): string;

  // Used for displaying the field in the UI
  abstract get label(): string;

  aliasedAs(alias: string) {
    return new DisplayField.Aliased(this, alias);
  }

  dealias(): DisplayField {
    return this instanceof DisplayField.Aliased ? this.field : this;
  }

  static fromString(s: string) {
    return new DisplayField.Static(Field.fromString(s));
  }

  abstract clone(): DisplayField;

  toJSON() {
    return {
      key: this.key,
      name: this.name,
      label: this.label,
    };
  }
}

export namespace DisplayField {
  export class Static extends DisplayField {
    constructor(public field: Field) {
      super();
    }

    override clone() {
      return new DisplayField.Static(this.field.clone());
    }

    override get key() {
      if (this.field instanceof Field.Aliased) return this.field.alias;
      return this.field.toString();
    }

    override get name() {
      return this.field.displayFieldName;
    }

    override get label() {
      return toSentenceCase(this.key);
    }
  }

  export class Function extends DisplayField {
    constructor(
      public fnName: string,
      public args: string[],
    ) {
      super();
    }

    override clone() {
      return new DisplayField.Function(this.fnName, this.args);
    }

    override get key() {
      return `${this.name}_${this.args.join('_')}`;
    }

    override get name() {
      return { labels: 'labels' }[this.fnName] || this.fnName;
    }

    override get label() {
      return `${toSentenceCase(this.fnName)}: ${this.args.map(toSentenceCase).join(', ')}`;
    }
  }

  export class Aliased extends DisplayField {
    constructor(
      public field: DisplayField,
      public alias: string,
    ) {
      super();
    }

    override clone() {
      return new DisplayField.Aliased(this.field.clone(), this.alias);
    }

    override get key() {
      return this.field.key;
    }

    override get name() {
      return this.field.dealias().name;
    }

    override get label() {
      return this.alias;
    }
  }
}
