import { uniqueId } from '../utils/common';
import { TimeRange } from '../utils/date_time/time_range';
import { Dimension } from './dimension';
import { DisplayField } from './display_field';
import { Operator } from './operator';
import { Query } from './query';
import { Value } from './value';

export const enum DimensionSegmentType {
  Time = 'time',
  User = 'user',
  Unknown = 'unknown',
}

const knownSegments: Record<string, DimensionSegment> = {};

function toDateTimeString(date: Date) {
  return date.toISOString().replace('T', ' ').split(':').slice(0, -1).join(':');
}

export abstract class DimensionSegment {
  public abstract type: DimensionSegmentType;
  public key: string;

  abstract get title(): string;

  constructor(public dimension: Dimension) {
    this.key = uniqueId('s');
    knownSegments[this.key] = this;
  }

  static fromKey(key: string): DimensionSegment | void {
    return knownSegments[key];
  }

  abstract apply(query: Query): void;

  static combine(
    a: DimensionSegment[],
    b?: DimensionSegment[],
  ): [DimensionSegment, DimensionSegment][] | [DimensionSegment][] {
    if (!b) return a.map((a) => [a]);

    return a.flatMap((a) => b.map((b) => [a, b] as [DimensionSegment, DimensionSegment]));
  }
}

export namespace DimensionSegment {
  export class Time extends DimensionSegment {
    public type = DimensionSegmentType.Time;

    constructor(
      dimension: Dimension,
      public range: TimeRange,
    ) {
      super(dimension);
    }

    override get title() {
      return this.range.toString();
    }

    override apply(query: Query): void {
      const dimensionField = this.dimension.field.dealias();
      if (!(dimensionField instanceof DisplayField.Static)) return;

      query.setValue(
        dimensionField.field,
        Operator.LessThan,
        new Value.Quoted(toDateTimeString(this.range.to)),
      );
      query.setValue(
        dimensionField.field,
        Operator.GreaterThan,
        new Value.Quoted(toDateTimeString(this.range.from)),
      );
    }
  }

  export class User extends DimensionSegment {
    public type = DimensionSegmentType.User;

    constructor(
      dimension: Dimension,
      public user: Value.Reference.User,
    ) {
      super(dimension);
    }

    override get title() {
      return this.user.toString();
    }

    override apply(query: Query): void {
      const dimensionField = this.dimension.field.dealias();
      if (!(dimensionField instanceof DisplayField.Static)) return;

      query.setValue(dimensionField.field, Operator.Equal, this.user);
    }
  }

  export class Unknown extends DimensionSegment {
    public type = DimensionSegmentType.Unknown;

    override apply(): void {
      throw new Error('Unknown dimension segment cannot be applied');
    }

    override get title() {
      return 'Unknown';
    }
  }
}
