import { TimeSegmentType } from '../utils/date_time/time_segment_type';
import { TimeUnit } from '../utils/date_time/time_unit';
import { DimensionSegment, DimensionSegmentType } from './dimension_segment';
import { Value } from './value';
import { TimeRange } from '../utils/date_time/time_range';
import { Dimension } from './dimension';
import { Query } from './query';
import { DisplayField } from './display_field';
import { Operator } from './operator';

export abstract class DimensionFunction {
  public abstract type: DimensionSegmentType;

  abstract segment(dimension: Dimension, query: Query): DimensionSegment[];
}

export namespace DimensionFunction {
  export class Time extends DimensionFunction {
    public type = DimensionSegmentType.Time;

    constructor(
      public quantity: number,
      public unit: TimeUnit,
      public timeSegmentType: TimeSegmentType,
    ) {
      super();
    }

    override segment(dimension: Dimension, query: Query): DimensionSegment.Time[] {
      const dimensionField = dimension.field.dealias();
      if (!(dimensionField instanceof DisplayField.Static))
        throw new Error('Dimension field is not static');

      const field = dimensionField.field.dealias();
      const before = query.getValue(field, Operator.LessThanEquals, Operator.LessThan);
      const after = query.getValue(field, Operator.GreaterThanEquals, Operator.GreaterThan);

      if (!(before instanceof Value.Date && after instanceof Value.Date))
        throw new Error('Before and after values are not dates');

      return TimeRange.fromString(after.value, before.value)
        .segment(this.quantity, this.unit, this.timeSegmentType)
        .map((range) => new DimensionSegment.Time(dimension, range));
    }
  }

  export class User extends DimensionFunction {
    public type = DimensionSegmentType.User;

    constructor(public users: Value.Reference.User[]) {
      super();
    }

    override segment(dimension: Dimension): DimensionSegment.User[] {
      return this.users.map((user) => new DimensionSegment.User(dimension, user));
    }
  }

  export class Unknown extends DimensionFunction {
    public type = DimensionSegmentType.Unknown;

    override segment(): DimensionSegment[] {
      return [];
    }
  }
}
