import { Query } from '../types/query';
import { Context } from '../types/context';
import { Value } from '../types/value';
import { calculateDays, startOfDay } from '../utils/date_time';
import { GlqlAnalyzeError } from '../errors';
import { ReferenceType } from '../types/reference_type';

export function applyFunctions(query: Query) {
  query.expressions.forEach((expr) => {
    expr.value = processValue(expr.value, query.context);
  });
}

function processValue(value: Value, context: Context): Value {
  if (value instanceof Value.Function) {
    return evaluateFunction(value.value, value.args, context);
  }

  if (value instanceof Value.Date) {
    return new Value.Quoted(validateDateFormat(value.value));
  }

  if (value instanceof Value.RelativeDate) {
    const { value: quantity, unit } = value;
    const days = calculateDays(quantity, unit);
    return new Value.Quoted(startOfDay(days));
  }

  if (value instanceof Value.List) {
    return new Value.List(value.value.map((v) => processValue(v, context)));
  }

  return value;
}

function validateDateFormat(date: string): string {
  const dateObj = new Date(date);
  if (isNaN(dateObj.getTime())) {
    throw new GlqlAnalyzeError.InvalidDateFormat(date);
  }
  return date;
}

function evaluateFunction(funcName: string, funcArgs: string[], context: Context): Value {
  switch (funcName.toLowerCase()) {
    case 'today':
      return new Value.Quoted(startOfDay(0));
    case 'startofday':
      try {
        const day = parseInt(funcArgs[0]!);
        return new Value.Quoted(startOfDay(day));
      } catch {
        throw new GlqlAnalyzeError.UnknownError();
      }
    case 'currentuser':
      return context.username
        ? new Value.Reference(ReferenceType.User, context.username)
        : new Value.Null();
    default:
      throw new GlqlAnalyzeError.UnrecognizedFunction(funcName, funcArgs);
  }
}
