import { transformFields } from './field_functions';
import { DisplayField } from '../types/display_field';
import { Field, FieldName } from '../types/field';

describe('field_functions', () => {
  describe('transform_fields', () => {
    it('should transform fields correctly', () => {
      const fields = [
        new DisplayField.Static(new Field(FieldName.Id)),
        new DisplayField.Function('labels', ['bug', 'critical']),
        new DisplayField.Function('labels', ['feature*', '*maintenance*']).aliasedAs(
          'Feature/Maintenance',
        ),
      ];

      const mockIssues = {
        nodes: [
          {
            id: '1',
            title: 'Issue 1',
            labels: {
              nodes: [
                { title: 'bug::closed' },
                { title: 'bug' },
                { title: 'critical' },
                { title: 'feature-request' },
              ],
            },
          },
          {
            id: '2',
            title: 'Issue 2',
            labels: {
              nodes: [
                { title: 'feature-request' },
                { title: 'enhancement' },
                { title: 'a-maintenance-b' },
                { title: 'documentation' },
              ],
            },
          },
        ],
        count: 100,
      };

      const result = transformFields(mockIssues, fields);
      const expected = {
        nodes: [
          {
            id: '1',
            title: 'Issue 1',
            labels_bug_critical: {
              nodes: [{ title: 'bug' }, { title: 'critical' }],
            },
            'labels_feature*_*maintenance*': {
              nodes: [{ title: 'feature-request' }],
            },
            labels: {
              nodes: [{ title: 'bug::closed' }],
            },
          },
          {
            id: '2',
            title: 'Issue 2',
            labels_bug_critical: {
              nodes: [],
            },
            'labels_feature*_*maintenance*': {
              nodes: [{ title: 'feature-request' }, { title: 'a-maintenance-b' }],
            },
            labels: {
              nodes: [{ title: 'enhancement' }, { title: 'documentation' }],
            },
          },
        ],
        count: 100,
      };

      expect(result).toEqual(expected);
    });

    it('should handle derived fields and aliases', () => {
      const mockData = {
        nodes: [
          {
            id: '1',
            title: 'Issue with login',
            createdAt: '2024-01-15T10:30:00Z',
            lastComment: {
              nodes: [{ bodyHtml: 'This is the last comment' }],
            },
          },
          {
            id: '2',
            title: 'Feature request',
            createdAt: '2024-01-16T14:20:00Z',
            lastComment: {
              nodes: [],
            },
          },
        ],
        count: 100,
      };

      // Test fields with aliases: createdAt -> created, lastComment -> lastComment
      const fields = [
        new DisplayField.Static(new Field(FieldName.Id)),
        new DisplayField.Static(new Field(FieldName.Title)),
        new DisplayField.Static(new Field(FieldName.LastComment)),
        new DisplayField.Static(new Field(FieldName.Created)),
      ];

      const result = transformFields(mockData, fields);
      const expected = {
        nodes: [
          {
            id: '1',
            title: 'Issue with login',
            createdAt: '2024-01-15T10:30:00Z',
            created: '2024-01-15T10:30:00Z',
            lastComment: 'This is the last comment',
          },
          {
            id: '2',
            title: 'Feature request',
            createdAt: '2024-01-16T14:20:00Z',
            created: '2024-01-16T14:20:00Z',
            lastComment: null,
          },
        ],
        count: 100,
      };

      expect(result).toEqual(expected);
    });
  });
});
