import { Expression } from '../types/expression';
import { Value } from '../types/value';
import { GlqlAnalyzeError } from '../errors';
import { Source } from '../types/source';
import { Operator, OperatorType } from '../types/operator';

export function combineExpressions(expressions: Expression[]): Expression[] {
  const grouped: Record<string, Expression[]> = {};
  const hash = (expr: Expression) => `${expr.field}${expr.operator}`;

  for (const expr of expressions) {
    (grouped[hash(expr)] ||= []).push(expr);
  }

  return Object.entries(grouped).map(([, exprs]) => {
    return new Expression(
      exprs[0]!.field,
      exprs[0]!.operator,
      exprs.reduce((acc, { value }) => combineValues(acc, value), exprs[0]!.value),
    );
  });
}

function combineValues(a: Value, b: Value): Value {
  if (a.eq(b)) return a;

  if (
    (a instanceof Value.Quoted || a instanceof Value.Reference) &&
    (b instanceof Value.Quoted || b instanceof Value.Reference)
  ) {
    return a.value === b.value ? a : new Value.List([a, b]);
  }

  if (a instanceof Value.List && b instanceof Value.List) {
    const combinedValues = [...a.value, ...b.value];
    const uniqueValues: Value[] = [];

    for (const value of combinedValues) {
      if (!uniqueValues.some((existing) => existing.eq(value))) {
        uniqueValues.push(value);
      }
    }

    return new Value.List(uniqueValues);
  }

  if (a instanceof Value.List && (b instanceof Value.Quoted || b instanceof Value.Reference)) {
    return new Value.List([...a.value, b]);
  }

  if ((a instanceof Value.Quoted || a instanceof Value.Reference) && b instanceof Value.List) {
    return new Value.List([a, ...b.value]);
  }

  throw new GlqlAnalyzeError.CannotCombineValues(a, b);
}

export function expandExpression(expr: Expression, source: Source): Expression[] {
  if (
    source.analyzer.fieldType(expr.field).isDateLike &&
    expr.operator.type === OperatorType.Equal &&
    isDateCastable(expr.value)
  ) {
    return [
      new Expression(expr.field, Operator.GreaterThanEquals, expr.value),
      new Expression(expr.field, Operator.LessThanEquals, expr.value),
    ];
  }

  return [expr];
}

function isFalsey(v: Value) {
  if (v instanceof Value.Null || (v instanceof Value.Bool && v.value === false)) {
    return true;
  }

  if (v instanceof Value.Number && v.value === 0) {
    return true;
  }

  if (v instanceof Value.Quoted && v.value.toLowerCase() === 'false') {
    return true;
  }

  return false;
}

function transformDate(v: Value, o: Operator) {
  if (v instanceof Value.Date || v instanceof Value.Quoted) {
    switch (o.type) {
      case OperatorType.GreaterThanEquals:
      case OperatorType.LessThan:
        return new Value.Date(`${v.value} 00:00`);
      case OperatorType.GreaterThan:
      case OperatorType.LessThanEquals:
        return new Value.Date(`${v.value} 23:59`);
    }
  }

  return v;
}

function isTruthy(v: Value) {
  return !isFalsey(v);
}

function truthy(x: Value): Value.Bool {
  return isTruthy(x) ? new Value.Bool(true) : new Value.Bool(false);
}

function isDateCastable(v: Value) {
  if (v instanceof Value.Date || v instanceof Value.RelativeDate) return true;
  if (v instanceof Value.Quoted) return !isNaN(new Date(v.value).getTime());

  return false;
}

// Convert all tokens to uppercase
export function transformTokensInValue(v: Value): Value {
  if (v instanceof Value.Token) return new Value.Token(v.value.toUpperCase());
  if (v instanceof Value.List) return new Value.List(v.value.map(transformTokensInValue));

  return v;
}

export function transformExpression(expr: Expression, source: Source): Expression {
  const f = expr.field.dealias();
  const o = expr.operator;
  const v = expr.value;

  const fieldType = source.analyzer.fieldType(f);
  const token = (v: string) => new Value.Token(v);

  if (o.type === OperatorType.NotEqual && v instanceof Value.List && v.value.length === 0) {
    return new Expression(f, Operator.Equal, token('ANY'));
  }

  if (o.type === OperatorType.Equal && v instanceof Value.List && v.value.length === 0) {
    return new Expression(f, Operator.Equal, token('NONE'));
  }

  if (o.type === OperatorType.NotEqual && v instanceof Value.Bool) {
    return new Expression(f, Operator.Equal, new Value.Bool(!v.value));
  }

  if (
    o.type === OperatorType.GreaterThan ||
    o.type === OperatorType.LessThan ||
    o.type === OperatorType.GreaterThanEquals ||
    o.type === OperatorType.LessThanEquals
  ) {
    return new Expression(f, o, transformDate(v, o));
  }

  if (v instanceof Value.Bool && fieldType.isBooleanLike) {
    return new Expression(f, o, v);
  }

  if (fieldType.isBooleanLike) {
    return new Expression(f, o, truthy(v));
  }

  if (o.type === OperatorType.In && fieldType.isHasOneListLike) {
    return new Expression(f, Operator.Equal, v);
  }

  if (
    o.type === OperatorType.NotEqual &&
    fieldType.isNullable &&
    v instanceof Value.Token &&
    v.value === 'ANY'
  ) {
    return new Expression(f, Operator.Equal, token('NONE'));
  }

  if (
    o.type === OperatorType.NotEqual &&
    fieldType.isNullable &&
    v instanceof Value.Token &&
    v.value === 'NONE'
  ) {
    return new Expression(f, Operator.Equal, token('ANY'));
  }

  return new Expression(f, o, v);
}
