import { GlqlAnalyzeError, InvalidScopeType } from '../errors';
import { featureFlags } from '../utils/feature_flags';
import { transformFields } from './field_functions';
import { DisplayField } from '../types/display_field';
import { SegmentedQuery } from '../types/segmented_query';

const DATA_SOURCES = ['epics', 'issues', 'mergeRequests', 'workItems'];

/// Transforms work items by merging widget data into the work item
/// and removing the widgets field
// eslint-disable-next-line @typescript-eslint/no-explicit-any
export function transformWorkItems(data: any) {
  const result = data;
  if (!result.nodes) return result;

  for (let i = 0; ; i++) {
    const workItem = result.nodes[i];
    if (!workItem) break;
    if (!workItem.widgets) continue;

    for (let j = 0; ; j++) {
      const widget = workItem.widgets[j];
      if (!widget) break;

      for (const [key, value] of Object.entries(widget)) {
        if (key !== 'type' && key !== '__typename') {
          result.nodes[i][key] = value;
        }
      }
    }

    delete result.nodes[i].widgets;
  }

  return result;
}

// eslint-disable-next-line @typescript-eslint/no-explicit-any
function transformAggregatedData(data: any) {
  const result = [];

  for (const [dimensionKey, metricsObj] of Object.entries(data)) {
    if (dimensionKey === '__typename') {
      continue;
    }

    if (!dimensionKey.startsWith('glql_')) {
      // skip the transformation if the data is not aggregated
      return data;
    }

    const segments = SegmentedQuery.segmentsFromKey(dimensionKey);

    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const item: any = {
      __typename: 'GlqlAggregatedData',
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      ...segments.reduce((acc: any, segment) => {
        acc[segment.dimension.field.name] = Object.assign(segment, {
          __typename: 'GlqlDimensionSegment',
        });
        return acc;
      }, {}),
    };

    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    for (const [metricKey, metricValue] of Object.entries(metricsObj as any)) {
      item[metricKey] = metricValue;
    }

    result.push(item);
  }

  return {
    __typename: 'GlqlAggregatedDataConnection',
    nodes: result,
    count: result.length,
  };
}

// eslint-disable-next-line @typescript-eslint/no-explicit-any
export function transformForDataSource(data: any) {
  const scope = data.project || data.group || {};

  if (Object.keys(scope).length === 0) {
    let invalidScopeType: InvalidScopeType;
    if ('project' in data && !data.project) {
      invalidScopeType = InvalidScopeType.Project;
    } else if ('group' in data && !data.group) {
      invalidScopeType = InvalidScopeType.Group;
    } else {
      invalidScopeType = InvalidScopeType.Namespace;
    }
    throw new GlqlAnalyzeError.InvalidScope(invalidScopeType);
  }

  for (const source of DATA_SOURCES) {
    if (scope[source]) {
      let transformed = scope[source];
      if (source === 'workItems') {
        transformed = transformWorkItems(transformed);
        featureFlags.glqlWorkItems = true;
      } else {
        featureFlags.glqlWorkItems = false;
      }
      return transformed;
    }
  }

  return scope;
}

export function transform(
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  data: any,
  fields: [DisplayField, ...DisplayField[]],
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
): any {
  data = structuredClone(data);
  data = transformForDataSource(data);
  data = transformAggregatedData(data);
  data = transformFields(data, fields);

  return data;
}
