import { transform, transformWorkItems, transformForDataSource } from './data';
import { DisplayField } from '../types/display_field';
import { Field, FieldName } from '../types/field';

describe('data', () => {
  // Mock data functions matching the JavaScript tests
  const mockLabels1 = () => ({ nodes: [{ title: 'bug' }] });
  const mockLabels2 = () => ({ nodes: [{ title: 'feature' }] });

  const mockIssues = () => ({
    issues: {
      nodes: [
        { id: '1', title: 'Lorem ipsum', labels: mockLabels1() },
        { id: '2', title: 'Dolor sit amet', labels: mockLabels2() },
      ],
    },
  });

  const mockEpics = () => ({
    epics: {
      nodes: [
        { id: '1', title: 'Lorem ipsum', labels: mockLabels1() },
        { id: '2', title: 'Dolor sit amet', labels: mockLabels2() },
      ],
    },
  });

  const mockMergeRequests = () => ({
    mergeRequests: {
      nodes: [
        { id: '1', title: 'Lorem ipsum', labels: mockLabels1() },
        { id: '2', title: 'Dolor sit amet', labels: mockLabels2() },
      ],
    },
  });

  const mockWorkItems = () => ({
    workItems: {
      nodes: [
        {
          id: '1',
          title: 'Lorem ipsum',
          widgets: [
            {},
            {},
            {},
            {
              __typename: 'WorkItemWidgetLabels',
              type: 'LABELS',
              labels: mockLabels1(),
            },
          ],
        },
        {
          id: '2',
          title: 'Dolor sit amet',
          widgets: [
            {},
            {},
            {},
            {
              __typename: 'WorkItemWidgetLabels',
              type: 'LABELS',
              labels: mockLabels2(),
            },
          ],
        },
      ],
    },
  });

  const mockWorkItemsWithoutWidgets = () => ({
    workItems: {
      nodes: [
        { id: '1', title: 'Lorem ipsum' },
        { id: '2', title: 'Dolor sit amet' },
      ],
    },
  });

  describe('transformForDataSource', () => {
    it('should transform issues data source', () => {
      const mockData = { project: mockIssues() };
      const result = transformForDataSource(mockData);

      expect(result).toEqual(mockIssues().issues);
    });

    it('should transform epics data source', () => {
      const mockData = { project: mockEpics() };
      const result = transformForDataSource(mockData);

      expect(result).toEqual(mockEpics().epics);
    });

    it('should transform merge requests data source', () => {
      const mockData = { project: mockMergeRequests() };
      const result = transformForDataSource(mockData);

      expect(result).toEqual(mockMergeRequests().mergeRequests);
    });

    it('should transform work items data source', () => {
      const mockData = { project: mockWorkItems() };
      const result = transformForDataSource(mockData);

      // Work items should be transformed (widgets merged and removed)
      const expected = {
        nodes: [
          {
            id: '1',
            title: 'Lorem ipsum',
            labels: mockLabels1(),
            widgets: undefined,
          },
          {
            id: '2',
            title: 'Dolor sit amet',
            labels: mockLabels2(),
            widgets: undefined,
          },
        ],
      };
      expect(result).toEqual(expected);
    });

    it('should handle group scope errors', () => {
      const mockData = { group: null };
      expect(() => transformForDataSource(mockData)).toThrow(
        'Group does not exist or you do not have access to it',
      );

      const mockData2 = { project: null };
      expect(() => transformForDataSource(mockData2)).toThrow(
        'Project does not exist or you do not have access to it',
      );
    });

    it('should handle no matching source', () => {
      const mockData = {
        project: {
          unknownSource: { nodes: [] },
        },
      };
      const result = transformForDataSource(mockData);
      expect(result).toEqual(mockData.project);
    });

    it('should handle no scope', () => {
      const mockData = { other: mockIssues() };
      expect(() => transformForDataSource(mockData)).toThrow(
        'Namespace does not exist or you do not have access to it',
      );
    });
  });

  describe('transformWorkItems', () => {
    it('should transform work items with widgets', () => {
      const workItems = mockWorkItems().workItems;
      const result = transformWorkItems(workItems);

      const expected = {
        nodes: [
          {
            id: '1',
            title: 'Lorem ipsum',
            labels: mockLabels1(),
            widgets: undefined,
          },
          {
            id: '2',
            title: 'Dolor sit amet',
            labels: mockLabels2(),
            widgets: undefined,
          },
        ],
      };
      expect(result).toEqual(expected);
    });

    it('should handle work items without widgets', () => {
      const workItems = mockWorkItemsWithoutWidgets().workItems;
      const result = transformWorkItems(workItems);

      // Should remain unchanged since there are no widgets
      const expected = {
        nodes: [
          { id: '1', title: 'Lorem ipsum', widgets: undefined },
          { id: '2', title: 'Dolor sit amet', widgets: undefined },
        ],
      };
      expect(result).toEqual(expected);
    });

    it('should handle empty widgets', () => {
      const workItems = {
        workItems: {
          nodes: [
            {
              id: '1',
              title: 'Lorem ipsum',
              widgets: [],
            },
          ],
        },
      };
      const workItemsData = workItems.workItems;
      const result = transformWorkItems(workItemsData);

      const expected = {
        nodes: [
          {
            id: '1',
            title: 'Lorem ipsum',
            widgets: undefined,
          },
        ],
      };
      expect(result).toEqual(expected);
    });

    it('should handle complex widgets', () => {
      const workItems = {
        workItems: {
          nodes: [
            {
              id: '1',
              title: 'Lorem ipsum',
              widgets: [
                {
                  __typename: 'WorkItemWidgetLabels',
                  type: 'LABELS',
                  labels: mockLabels1(),
                },
                {
                  __typename: 'WorkItemWidgetDescription',
                  type: 'DESCRIPTION',
                  description: 'Test description',
                },
                {
                  __typename: 'WorkItemWidgetAssignees',
                  type: 'ASSIGNEES',
                  assignees: { nodes: [{ name: 'John' }] },
                },
              ],
            },
          ],
        },
      };
      const workItemsData = workItems.workItems;
      const result = transformWorkItems(workItemsData);

      const expected = {
        nodes: [
          {
            id: '1',
            title: 'Lorem ipsum',
            labels: mockLabels1(),
            description: 'Test description',
            assignees: { nodes: [{ name: 'John' }] },
            widgets: undefined,
          },
        ],
      };
      expect(result).toEqual(expected);
    });

    it('should preserve existing fields', () => {
      const workItems = {
        workItems: {
          nodes: [
            {
              id: '1',
              title: 'Lorem ipsum',
              existingField: 'should be preserved',
              widgets: [
                {
                  __typename: 'WorkItemWidgetLabels',
                  type: 'LABELS',
                  labels: mockLabels1(),
                },
              ],
            },
          ],
        },
      };
      const workItemsData = workItems.workItems;
      const result = transformWorkItems(workItemsData);

      const expected = {
        nodes: [
          {
            id: '1',
            title: 'Lorem ipsum',
            existingField: 'should be preserved',
            labels: mockLabels1(),
            widgets: undefined,
          },
        ],
      };
      expect(result).toEqual(expected);
    });

    it('should handle malformed data', () => {
      // Test with missing nodes
      const workItems = {
        workItems: {
          otherField: 'value',
        },
      };
      const workItemsData = workItems.workItems;
      const result = transformWorkItems(workItemsData);
      expect(result).toEqual(workItems.workItems);

      // Test with non-array nodes
      const workItems2 = {
        workItems: {
          nodes: 'not an array',
        },
      };
      const workItemsData2 = workItems2.workItems;
      const result2 = transformWorkItems(workItemsData2);
      expect(result2).toEqual(workItems2.workItems);
    });
  });

  describe('transform', () => {
    it('should transform data with fields', () => {
      const mockData = { project: mockIssues() };
      const fields = [
        new DisplayField.Static(new Field(FieldName.Id)),
        new DisplayField.Static(new Field(FieldName.Title)),
      ] as [DisplayField, ...DisplayField[]];

      const result = transform(mockData, fields);

      // The transform function should call transformFields after data source transformation
      expect(result).toBeDefined();
      expect(result.nodes).toBeDefined();
    });
  });
});
