import { alt, regexp, seqMap, string, takeWhile, regex } from 'parsimmon';
import { Value } from '../types/value';
import { ReferenceType } from '../types/reference_type';
import { TimeUnit } from '../utils/date_time/time_unit';
import { GlqlAnalyzeError, GlqlParseError, GlqlParseErrorType } from '../errors';

// Parser for relative time
function relativeDateInner(allowNegative: boolean) {
  return seqMap(
    regexp(allowNegative ? /[+-]?/ : /\+?/),
    regexp(/\d+/),
    regexp(/\w/),
    (sign, number, unit) => {
      const signValue = sign === '-' ? -1 : 1;
      const timeUnit = (() => {
        switch (unit) {
          case 'd':
            return TimeUnit.Day;
          case 'w':
            return TimeUnit.Week;
          case 'm':
            return TimeUnit.Month;
          case 'y':
            return TimeUnit.Year;
          default:
            throw new GlqlParseError(GlqlParseErrorType.InvalidRelativeDateUnit, unit);
        }
      })();
      return new Value.RelativeDate(signValue * parseInt(number, 10), timeUnit);
    },
  );
}

export const relativeDate = relativeDateInner(true);

// Parser for dates
export const date = seqMap(
  regexp(/\d+/),
  regexp(/[-/.]/),
  regexp(/\d*/),
  regexp(/[-/.]?/),
  regexp(/\d*/),
  (yyyy, sep1, mm, sep2, dd) => {
    if (sep1 !== '-' || sep2 !== '-' || yyyy.length !== 4) {
      throw new GlqlAnalyzeError.InvalidDateFormat(`${yyyy}${sep1}${mm}${sep2}${dd}`);
    }
    return new Value.Date(`${yyyy}-${mm}-${dd}`);
  },
);

// Parser for bare words (tokens) - only valid tokens will parse
export const token = alt(
  regexp(
    /any|none|current|upcoming|started|all|opened|closed|merged|issue|mergerequest|incident|testcase|requirement|task|ticket|objective|keyresult|epic/i,
  ).map((value) => new Value.Token(value)),
);

// Parser for usernames
export const username = seqMap(
  string('@'),
  takeWhile((c) => !/[\s(),]/.test(c)).map((content) => {
    if (/^\d/.test(content) || !/^[a-zA-Z0-9_.-]*$/.test(content)) {
      throw new GlqlParseError(GlqlParseErrorType.InvalidUsername, `@${content}`);
    }
    return content;
  }),
  (_, content) => new Value.Reference(ReferenceType.User, content),
);

// Parser for milestone
export const milestone = seqMap(
  string('%'),
  alt(
    // Quoted string
    seqMap(
      string('"'),
      takeWhile((c) => c !== '"'),
      regexp(/"?/),
      (_, content, endQuote) => {
        if (endQuote !== '"' || !content) {
          throw new GlqlParseError(GlqlParseErrorType.InvalidMilestone, `"${content}`);
        }
        return content.trim();
      },
    ),
    // Bare word
    takeWhile((c) => !/[\s(),]/.test(c)).map((content) => {
      if (!/^[a-zA-Z0-9_.-]+$/.test(content)) {
        throw new GlqlParseError(GlqlParseErrorType.InvalidMilestone, `%${content}`);
      }
      return content;
    }),
  ),
  (_, content) => new Value.Reference(ReferenceType.Milestone, content),
);

// Parser for label
export const label = seqMap(
  string('~'),
  alt(
    // Quoted string
    seqMap(
      string('"'),
      takeWhile((c) => c !== '"'),
      regex(/"?/),
      (_, content, endQuote) => {
        if (endQuote !== '"') {
          throw new GlqlParseError(GlqlParseErrorType.InvalidLabel, `~"${content}`);
        }
        return content.trim();
      },
    ),
    // Bare word
    takeWhile((c) => /[a-zA-Z0-9_.:-]/.test(c)),
  ).assert((content) => Boolean(content && content.length > 0), 'Empty label'),
  (_, content) => new Value.Reference(ReferenceType.Label, content),
);

// Parser for iteration
export const iteration = seqMap(
  string('*iteration:'),
  takeWhile((c) => !/[\s(),]/.test(c)).map((content) => {
    if (!/^\d+$/.test(content)) {
      throw new GlqlParseError(GlqlParseErrorType.InvalidIteration, `*iteration:${content}`);
    }
    return content;
  }),
  (_, content) => new Value.Reference(ReferenceType.Iteration, content),
);

// Parser builder for work item references
export const workItem = (char: string, type: ReferenceType) =>
  seqMap(
    takeWhile((c) => /[a-zA-Z0-9/_.-]/.test(c)),
    string(char),
    takeWhile((c) => /[0-9]/.test(c)).assert(
      (content) => Boolean(content && content.length > 0),
      'Empty work item ID',
    ),
    (path, _, id) => {
      const reference = path ? `${path}${char}${id}` : id;
      return new Value.Reference(type, reference);
    },
  );

// Parser for epic references
// Format: [group_path]&<epic_id>
export const epic = workItem('&', ReferenceType.Epic);

// Parser for issue references
// Format: [group/project_path]#<issue_id>
export const issue = workItem('#', ReferenceType.WorkItem);
