import { relativeDate, date, token, username, milestone, label, iteration, epic } from './special';
import { Value } from '../types/value';
import { ReferenceType } from '../types/reference_type';
import { TimeUnit } from '../utils/date_time/time_unit';

describe('Special Parser', () => {
  describe('relativeDate', () => {
    it('should parse negative relative dates', () => {
      const result = relativeDate.parse('-7d');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.RelativeDate);
        expect(result.value.value).toBe(-7);
        expect(result.value.unit).toBe(TimeUnit.Day);
      }
    });

    it('should parse positive relative dates', () => {
      const result = relativeDate.parse('+30m');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.RelativeDate);
        expect(result.value.value).toBe(30);
        expect(result.value.unit).toBe(TimeUnit.Month);
      }
    });

    it('should parse dates without explicit sign', () => {
      const result = relativeDate.parse('12m');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.RelativeDate);
        expect(result.value.value).toBe(12);
        expect(result.value.unit).toBe(TimeUnit.Month);
      }
    });

    it('should fail on invalid units', () => {
      expect(() => relativeDate.parse('7x')).toThrow();
    });
  });

  describe('date', () => {
    it('should parse valid date formats', () => {
      const result = date.parse('2023-05-01');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Date);
        expect(result.value.value).toBe('2023-05-01');
      }
    });

    it('should fail on invalid date formats', () => {
      expect(date.parse('2023_05_01').status).toBe(false);
      expect(date.parse('').status).toBe(false);
      expect(date.parse('invalid').status).toBe(false);
    });
  });

  describe('token', () => {
    it('should parse valid tokens', () => {
      const result = token.parse('ANY');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Token);
        expect(result.value.value).toBe('ANY');
      }
    });

    it('should parse tokens case-insensitively', () => {
      const result = token.parse('NoNe');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Token);
        expect(result.value.value).toBe('NoNe');
      }
    });

    it('should fail on invalid tokens', () => {
      expect(token.parse('invalid').status).toBe(false);
    });
  });

  describe('username', () => {
    it('should parse valid usernames', () => {
      const result = username.parse('@user123');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Reference);
        expect(result.value.referenceType).toBe(ReferenceType.User);
        expect(result.value.value).toBe('user123');
      }
    });

    it('should fail on usernames without @', () => {
      expect(username.parse('user123').status).toBe(false);
    });
  });

  describe('milestone', () => {
    it('should parse quoted milestones', () => {
      const result = milestone.parse('%"17.5"');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Reference);
        expect(result.value.referenceType).toBe(ReferenceType.Milestone);
        expect(result.value.value).toBe('17.5');
      }
    });

    it('should parse bare word milestones', () => {
      const result = milestone.parse('%Backlog');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Reference);
        expect(result.value.referenceType).toBe(ReferenceType.Milestone);
        expect(result.value.value).toBe('Backlog');
      }
    });

    it('should trim spaces in quoted milestones', () => {
      const result = milestone.parse('%"  trims spaces  "');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value.value).toBe('trims spaces');
      }
    });

    it('should fail on invalid milestones', () => {
      expect(milestone.parse('').status).toBe(false);
      expect(milestone.parse('17.5').status).toBe(false);

      expect(() => milestone.parse('%')).toThrow();
      expect(() => milestone.parse('%""')).toThrow();
    });
  });

  describe('label', () => {
    it('should parse quoted labels', () => {
      const result = label.parse('~"🌭"');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Reference);
        expect(result.value.referenceType).toBe(ReferenceType.Label);
        expect(result.value.value).toBe('🌭');
      }
    });

    it('should parse bare word labels', () => {
      const result = label.parse('~frontend');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Reference);
        expect(result.value.referenceType).toBe(ReferenceType.Label);
        expect(result.value.value).toBe('frontend');
      }
    });

    it('should parse labels with colons', () => {
      const result = label.parse('~"workflow::in review"');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value.value).toBe('workflow::in review');
      }
    });
  });

  describe('iteration', () => {
    it('should parse valid iterations', () => {
      const result = iteration.parse('*iteration:25263');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Reference);
        expect(result.value.referenceType).toBe(ReferenceType.Iteration);
        expect(result.value.value).toBe('25263');
      }
    });

    it('should fail on invalid iterations', () => {
      expect(iteration.parse('*iteration').status).toBe(false);

      expect(() => iteration.parse('*iteration:')).toThrow();
      expect(() => iteration.parse('*iteration:abc')).toThrow();
    });
  });

  describe('epic', () => {
    it('should parse epics with group path', () => {
      const result = epic.parse('path/to/project-with_hyphens-underscores.and.dots&123');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Reference);
        expect(result.value.referenceType).toBe(ReferenceType.Epic);
        expect(result.value.value).toBe('path/to/project-with_hyphens-underscores.and.dots&123');
      }
    });

    it('should parse epics without group path', () => {
      const result = epic.parse('&123');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Reference);
        expect(result.value.referenceType).toBe(ReferenceType.Epic);
        expect(result.value.value).toBe('123');
      }
    });

    it('should fail on invalid epics', () => {
      expect(epic.parse('').status).toBe(false);
      expect(epic.parse('&').status).toBe(false);
      expect(epic.parse('123').status).toBe(false);
    });
  });
});
