import { string, alt, regexp, seqMap, sepBy, optWhitespace, seq, Parser, regex } from 'parsimmon';
import { Value } from '../types/value';
import {
  date,
  epic,
  issue,
  iteration,
  label,
  milestone,
  relativeDate,
  token,
  username,
} from './special';
import { GlqlParseError, GlqlParseErrorType } from '../errors';

// Parser for null
export const nil = regexp(/null/i).map(() => new Value.Null());

// Parser for boolean
export const bool = alt(
  regexp(/true/i).map(() => new Value.Bool(true)),
  regexp(/false/i).map(() => new Value.Bool(false)),
);

// Parser for numbers
export const uint = regexp(/\d+/).map((s) => new Value.Number(parseInt(s, 10)));

// Helper function for string parsing
function createStringParser(quote: string) {
  return seqMap(
    string(quote),
    regexp(new RegExp(`[^${quote}\\\\]*(?:\\\\.[^${quote}\\\\]*)*`)),
    regex(new RegExp(`${quote}?`)),
    (_, content, endQuote) => {
      if (endQuote !== quote) {
        throw new GlqlParseError(GlqlParseErrorType.UnterminatedString, quote + content);
      }
      return new Value.Quoted(content.trim());
    },
  );
}

// Parser for strings (both single and double quotes)
export const str = alt(createStringParser('"'), createStringParser("'"));

// Parser for functions
export const fn = seqMap(
  regexp(/[a-zA-Z_][a-zA-Z0-9_]*/),
  optWhitespace,
  string('('),
  optWhitespace,
  alt(
    sepBy(
      regexp(/[^,)]+/).map((s) => s.trim()),
      string(','),
    ).map((values) => values.map((v) => v.replace(/^["']|["']$/g, ''))),
    optWhitespace.map(() => []),
  ),
  optWhitespace,
  string(')'),
  (name, _, __, ___, args) => new Value.Function(name, args),
);

// Define nonArrayValue here to avoid circular dependency
export const nonArrayValue: Parser<Value> = alt(
  nil,
  bool,
  relativeDate,
  date,
  username,
  milestone,
  label,
  iteration,
  epic,
  issue,
  str,
  uint,
  fn,
  token,
);

// Parser for arrays
export const array = seqMap(
  string('('),
  optWhitespace,
  sepBy(nonArrayValue, seq(optWhitespace, string(','), optWhitespace)),
  optWhitespace,
  string(')'),
  (_, __, values) => new Value.List(values),
);
