import { nil, bool, uint, str, array, fn } from './literals';
import { Value } from '../types/value';

describe('Literals Parser', () => {
  describe('nil', () => {
    it('should parse null', () => {
      const result = nil.parse('null');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Null);
      }
    });

    it('should fail on invalid input', () => {
      const result = nil.parse('notnull');
      expect(result.status).toBe(false);
    });
  });

  describe('bool', () => {
    it('should parse true', () => {
      const result = bool.parse('true');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Bool);
        expect(result.value.value).toBe(true);
      }
    });

    it('should parse false', () => {
      const result = bool.parse('false');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Bool);
        expect(result.value.value).toBe(false);
      }
    });

    it('should fail on invalid input', () => {
      const result = bool.parse('notbool');
      expect(result.status).toBe(false);
    });
  });

  describe('uint', () => {
    it('should parse numbers', () => {
      const result = uint.parse('123');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Number);
        expect(result.value.value).toBe(123);
      }
    });

    it('should fail on non-numeric input', () => {
      const result = uint.parse('abc');
      expect(result.status).toBe(false);
    });
  });

  describe('str', () => {
    it('should parse double-quoted strings', () => {
      const result = str.parse('"hello"');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Quoted);
        expect(result.value.value).toBe('hello');
      }
    });

    it('should parse single-quoted strings', () => {
      const result = str.parse("'world'");
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Quoted);
        expect(result.value.value).toBe('world');
      }
    });

    it('should trim whitespace', () => {
      const result = str.parse('"  trimmed  "');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value.value).toBe('trimmed');
      }
    });
  });

  describe('array', () => {
    it('should parse arrays with quoted strings', () => {
      const result = array.parse('("a", "b")');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.List);
        expect(result.value.value).toHaveLength(2);
        if (result.value.value[0] && result.value.value[1]) {
          expect(result.value.value[0].value).toBe('a');
          expect(result.value.value[1].value).toBe('b');
        }
      }
    });

    it('should parse arrays with mixed value types', () => {
      const result = array.parse('("hello", 123, true, null)');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.List);
        expect(result.value.value).toHaveLength(4);
        expect(result.value.value[0]).toBeInstanceOf(Value.Quoted);
        expect(result.value.value[1]).toBeInstanceOf(Value.Number);
        expect(result.value.value[2]).toBeInstanceOf(Value.Bool);
        expect(result.value.value[3]).toBeInstanceOf(Value.Null);
      }
    });

    it('should parse arrays with usernames', () => {
      const result = array.parse('(@john, @jane, "admin")');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.List);
        expect(result.value.value).toHaveLength(3);
      }
    });

    it('should parse arrays with labels', () => {
      const result = array.parse('(~backend, ~frontend, "urgent")');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.List);
        expect(result.value.value).toHaveLength(3);
      }
    });

    it('should parse arrays with milestones', () => {
      const result = array.parse('(%v1.0, %v2.0, "completed")');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.List);
        expect(result.value.value).toHaveLength(3);
      }
    });

    it('should parse arrays with iterations', () => {
      const result = array.parse('(*iteration:123, *iteration:456, "current")');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.List);
        expect(result.value.value).toHaveLength(3);
      }
    });

    it('should parse arrays with epics', () => {
      const result = array.parse('(project&123, project&456, "feature")');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.List);
        expect(result.value.value).toHaveLength(3);
      }
    });

    it('should parse arrays with functions', () => {
      const result = array.parse('(currentUser(), today(), "static")');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.List);
        expect(result.value.value).toHaveLength(3);
      }
    });

    it('should parse arrays with relative dates', () => {
      const result = array.parse('(yesterday(), today(), "2024-01-01")');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.List);
        expect(result.value.value).toHaveLength(3);
      }
    });

    it('should handle whitespace variations', () => {
      const result = array.parse('( "a" , "b" , "c" )');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.List);
        expect(result.value.value).toHaveLength(3);
      }
    });

    it('should handle empty arrays', () => {
      const result = array.parse('()');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.List);
        expect(result.value.value).toHaveLength(0);
      }
    });

    it('should fail on malformed arrays', () => {
      const result = array.parse('(1, 2,)');
      expect(result.status).toBe(false);
    });
  });

  describe('fn', () => {
    it('should parse functions without arguments', () => {
      const result = fn.parse('currentuser()');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Function);
        expect(result.value.value).toBe('currentuser');
        expect(result.value.args).toHaveLength(0);
      }
    });

    it('should parse functions with arguments', () => {
      const result = fn.parse('func("arg1", "arg2")');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Function);
        expect(result.value.value).toBe('func');
        expect(result.value.args).toHaveLength(2);
        expect(result.value.args[0]).toBe('arg1');
        expect(result.value.args[1]).toBe('arg2');
      }
    });

    it.each`
      input
      ${'func("arg1"  , "arg2")'}
      ${'func( "arg1", "arg2" )'}
      ${'func  ( "arg1" , "arg2" )'}
    `('should parse functions with whitespace around arguments: $input', ({ input }) => {
      const result = fn.parse(input);
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Function);
        expect(result.value.value).toBe('func');
        expect(result.value.args).toHaveLength(2);
        expect(result.value.args[0]).toBe('arg1');
        expect(result.value.args[1]).toBe('arg2');
      }
    });

    it.each`
      input
      ${'func ()'}
      ${'func( )'}
      ${'func  (    )'}
    `('should parse functions with whitespace without arguments: $input', ({ input }) => {
      const result = fn.parse(input);
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Value.Function);
        expect(result.value.value).toBe('func');
        expect(result.value.args).toHaveLength(0);
      }
    });
  });
});
