import { seqMap, optWhitespace, sepBy, seq, regexp, fail, succeed } from 'parsimmon';
import { expression } from './expression';
import { GlqlParseError, GlqlParseErrorType } from '../errors';
import { Expression } from '../types/expression';

// Parser for the entire query (multiple expressions separated by AND)
export const expressions = seqMap(
  optWhitespace,
  sepBy(expression, seq(optWhitespace, regexp(/AND/i), optWhitespace)),
  optWhitespace,
  (_, expressions) => expressions,
).chain((expressions) => {
  // Reject if no expressions
  if (expressions.length === 0) {
    return fail('Query must contain at least one expression');
  }
  return succeed(expressions);
});

export function parseExpressions(q: string): Expression[] {
  const result = expressions.parse(q);
  if (result.status) return result.value;

  // Handle parsing errors
  if (result.expected.length > 0) {
    const index = result.index.offset;
    const remaining = q.slice(index);

    if (remaining.trim() === '') {
      throw new GlqlParseError(GlqlParseErrorType.MissingValue, '');
    }

    throw new GlqlParseError(GlqlParseErrorType.UnexpectedToken, remaining);
  }

  // Fallback for unknown errors
  throw new GlqlParseError(GlqlParseErrorType.UnknownError, q);
}
