import { expressions } from './index';
import { Expression } from '../types/expression';
import { Operator } from '../types/operator';
import { Value } from '../types/value';

describe('Query Parser', () => {
  describe('expressions', () => {
    it('should parse single expression', () => {
      const result = expressions.parse('label = "backend"');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toHaveLength(1);
        expect(result.value[0]!).toBeInstanceOf(Expression);
        expect(result.value[0]!.field.toString()).toBe('label');
        expect(result.value[0]!.operator).toBe(Operator.Equal);
        expect(result.value[0]!.value).toBeInstanceOf(Value.Quoted);
      }
    });

    it('should parse two expressions with AND', () => {
      const result = expressions.parse('label = "backend" AND weight > 5');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toHaveLength(2);

        // First expression: label = "backend"
        expect(result.value[0]!.field.toString()).toBe('label');
        expect(result.value[0]!.operator).toBe(Operator.Equal);
        expect(result.value[0]!.value).toBeInstanceOf(Value.Quoted);

        // Second expression: weight > 5
        expect(result.value[1]!.field.toString()).toBe('weight');
        expect(result.value[1]!.operator).toBe(Operator.GreaterThan);
        expect(result.value[1]!.value).toBeInstanceOf(Value.Number);
      }
    });

    it('should parse multiple expressions with AND', () => {
      const result = expressions.parse('status = "open" AND weight = "high" AND author = "john"');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toHaveLength(3);

        expect(result.value[0]!.field.toString()).toBe('status');
        expect(result.value[1]!.field.toString()).toBe('weight');
        expect(result.value[2]!.field.toString()).toBe('author');
      }
    });

    it('should handle whitespace around AND', () => {
      const result = expressions.parse('label="backend"   AND   weight>5');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toHaveLength(2);
        expect(result.value[0]!.field.toString()).toBe('label');
        expect(result.value[1]!.field.toString()).toBe('weight');
      }
    });

    it('should handle case insensitive AND', () => {
      const result = expressions.parse('label = "backend" and weight > 5');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toHaveLength(2);
      }
    });

    it('should fail on invalid expression', () => {
      expect(() => expressions.parse('label = "backend" AND invalid syntax')).toThrow();
    });

    it('should fail on empty input', () => {
      expect(() => expressions.parse('')).toThrow();
    });

    it('should fail on just whitespace', () => {
      expect(() => expressions.parse('   ')).toThrow();
    });

    it('should fail on expression ending with AND', () => {
      expect(() => expressions.parse('label = "backend" AND')).toThrow();
    });

    it('should fail on expression starting with AND', () => {
      expect(() => expressions.parse('AND label = "backend"')).toThrow();
    });
  });
});
