import { alt, seqMap, string, regexp, optWhitespace, sepBy, seq } from 'parsimmon';
import { DisplayField } from '../types/display_field';
import { field } from './expression';
import { fn } from './literals';
import { Value } from '../types/value';

// Parser for static fields
const staticField = field.map((field) => new DisplayField.Static(field));

// Parser for field functions
const fieldFunction = fn.map((value) => {
  if (value instanceof Value.Function) {
    return new DisplayField.Function(value.value, value.args);
  }
  throw new Error('Expected function value');
});

// Parser for display fields (static or function)
const displayField = alt(fieldFunction, staticField);

// Parser for aliases (AS "alias")
const alias = seqMap(
  optWhitespace,
  regexp(/as/i),
  optWhitespace,
  alt(
    regexp(/"[^"]*"/).map((s) => s.slice(1, -1)), // Double quotes
    regexp(/'[^']*'/).map((s) => s.slice(1, -1)), // Single quotes
  ),
  (_, __, ___, alias) => alias,
);

// Parser for display fields with optional aliases
const displayFieldWithAlias = alt(
  seqMap(displayField, alias, (field, alias) => field.aliasedAs(alias)),
  displayField,
);

// Parser for field lists (comma-separated)
const fieldList = sepBy(
  displayFieldWithAlias,
  seq(optWhitespace, string(','), optWhitespace),
).assert((fields) => fields.length > 0, 'At least one field is required');

// Main parser for fields
export const parseFields = (input: string): DisplayField[] => {
  const result = fieldList.parse(input.trim());
  if (result.status) {
    return result.value;
  }
  throw new Error(`Failed to parse fields: ${result.expected.join(', ')}`);
};
