import { parseFields } from './fields';
import { DisplayField } from '../types/display_field';

describe('Fields Parser', () => {
  describe('parse_single_field', () => {
    it('should parse single field', () => {
      const result = parseFields('title');
      expect(result).toHaveLength(1);
      expect(result[0]).toBeInstanceOf(DisplayField.Static);
      expect(result[0]!.name).toBe('title');
    });

    it('should parse single field with whitespace', () => {
      const result = parseFields('  title  ');
      expect(result).toHaveLength(1);
      expect(result[0]).toBeInstanceOf(DisplayField.Static);
      expect(result[0]!.name).toBe('title');
    });
  });

  describe('parse_multiple_fields', () => {
    it('should parse multiple fields', () => {
      const result = parseFields('title,description,createdAt');
      expect(result).toHaveLength(3);

      expect(result[0]).toBeInstanceOf(DisplayField.Static);
      expect(result[0]!.name).toBe('title');

      expect(result[1]).toBeInstanceOf(DisplayField.Static);
      expect(result[1]!.name).toBe('descriptionHtml');

      expect(result[2]).toBeInstanceOf(DisplayField.Static);
      expect(result[2]!.name).toBe('createdAt');
      expect(result[2]!.key).toBe('createdAt');
    });
  });

  describe('parse_function_call', () => {
    it('should parse function call', () => {
      const result = parseFields("labels('bug')");
      expect(result).toHaveLength(1);
      expect(result[0]).toBeInstanceOf(DisplayField.Function);
      const func = result[0]! as DisplayField.Function;
      expect(func.fnName).toBe('labels');
      expect(func.args).toEqual(['bug']);
    });

    it('should parse function call with multiple arguments', () => {
      const result = parseFields("labels('bug', 'feature')");
      expect(result).toHaveLength(1);
      expect(result[0]).toBeInstanceOf(DisplayField.Function);
      const func = result[0]! as DisplayField.Function;
      expect(func.fnName).toBe('labels');
      expect(func.args).toEqual(['bug', 'feature']);
    });
  });

  describe('parse_mix_of_field_names_and_function_calls', () => {
    it('should parse mix of fields and functions', () => {
      const result = parseFields("title,labels('bug'),description");
      expect(result).toHaveLength(3);

      expect(result[0]).toBeInstanceOf(DisplayField.Static);
      expect(result[0]!.name).toBe('title');

      expect(result[1]).toBeInstanceOf(DisplayField.Function);
      const func1 = result[1]! as DisplayField.Function;
      expect(func1.fnName).toBe('labels');
      expect(func1.args).toEqual(['bug']);

      expect(result[2]).toBeInstanceOf(DisplayField.Static);
      expect(result[2]!.name).toBe('descriptionHtml');
    });
  });

  describe('handle_whitespace', () => {
    it('should handle whitespace around fields and commas', () => {
      const result = parseFields(' title , labels( "bug" ) , description ');
      expect(result).toHaveLength(3);

      expect(result[0]).toBeInstanceOf(DisplayField.Static);
      expect(result[0]!.name).toBe('title');

      expect(result[1]).toBeInstanceOf(DisplayField.Function);
      const func2 = result[1]! as DisplayField.Function;
      expect(func2.fnName).toBe('labels');
      expect(func2.args).toEqual(['bug']);

      expect(result[2]).toBeInstanceOf(DisplayField.Static);
      expect(result[2]!.name).toBe('descriptionHtml');
    });
  });

  describe('handle_alias', () => {
    it('should handle aliases for fields and functions', () => {
      const result = parseFields('title as "t", labels("bug") as "l", description as "d"');
      expect(result).toHaveLength(3);

      expect(result[0]).toBeInstanceOf(DisplayField.Aliased);
      expect((result[0] as DisplayField.Aliased).alias).toBe('t');
      const aliased1 = result[0]! as DisplayField.Aliased;
      expect(aliased1.field.name).toBe('title');

      expect(result[1]).toBeInstanceOf(DisplayField.Aliased);
      expect((result[1] as DisplayField.Aliased).alias).toBe('l');
      const aliased2 = result[1]! as DisplayField.Aliased;
      const func3 = aliased2.field as DisplayField.Function;
      expect(func3.fnName).toBe('labels');
      expect(func3.args).toEqual(['bug']);

      expect(result[2]).toBeInstanceOf(DisplayField.Aliased);
      expect((result[2] as DisplayField.Aliased).alias).toBe('d');
      const aliased3 = result[2]! as DisplayField.Aliased;
      expect(aliased3.field.name).toBe('descriptionHtml');
    });
  });

  describe('handle_case_sensitive_alias', () => {
    it('should handle case insensitive AS keyword', () => {
      const result = parseFields('title As "t", description, createdAt AS "c"');
      expect(result).toHaveLength(3);

      expect(result[0]).toBeInstanceOf(DisplayField.Aliased);
      expect((result[0] as DisplayField.Aliased).alias).toBe('t');
      const aliased4 = result[0]! as DisplayField.Aliased;
      expect(aliased4.field.name).toBe('title');

      expect(result[1]).toBeInstanceOf(DisplayField.Static);
      expect(result[1]!.name).toBe('descriptionHtml');

      expect(result[2]).toBeInstanceOf(DisplayField.Aliased);
      expect((result[2] as DisplayField.Aliased).alias).toBe('c');
      const aliased5 = result[2]! as DisplayField.Aliased;
      expect(aliased5.field.name).toBe('createdAt');
    });
  });

  describe('handle_alias_with_whitespace', () => {
    it('should handle aliases with various whitespace', () => {
      const result = parseFields(
        ' title  as  "t" , labels( "bug" )  as     "l" ,   description as  "d"  ',
      );
      expect(result).toHaveLength(3);

      expect(result[0]).toBeInstanceOf(DisplayField.Aliased);
      expect((result[0] as DisplayField.Aliased).alias).toBe('t');
      const aliased6 = result[0]! as DisplayField.Aliased;
      expect(aliased6.field.name).toBe('title');

      expect(result[1]).toBeInstanceOf(DisplayField.Aliased);
      expect((result[1] as DisplayField.Aliased).alias).toBe('l');
      const aliased7 = result[1]! as DisplayField.Aliased;
      const func4 = aliased7.field as DisplayField.Function;
      expect(func4.fnName).toBe('labels');
      expect(func4.args).toEqual(['bug']);

      expect(result[2]).toBeInstanceOf(DisplayField.Aliased);
      expect((result[2] as DisplayField.Aliased).alias).toBe('d');
      const aliased8 = result[2]! as DisplayField.Aliased;
      expect(aliased8.field.name).toBe('descriptionHtml');
    });
  });

  describe('error_scenarios', () => {
    it('should fail on incomplete alias', () => {
      expect(() => parseFields('title as')).toThrow();
    });

    it('should fail on invalid alias value', () => {
      expect(() => parseFields('title as 123')).toThrow();
    });

    it('should fail on trailing comma', () => {
      expect(() => parseFields('title,')).toThrow();
    });

    it('should fail on incomplete function call', () => {
      expect(() => parseFields("labels('bug'")).toThrow();
    });
  });
});
