import { alt, seqMap, string, regexp, optWhitespace, all } from 'parsimmon';
import { Field } from '../types/field';
import { Operator } from '../types/operator';
import { Expression } from '../types/expression';
import { value } from './value';
import { GlqlAnalyzeError, GlqlParseError, GlqlParseErrorType } from '../errors';
import { fn } from './literals';

// Parser for static fields (alpha characters)
const staticField = regexp(/[a-zA-Z_][a-zA-Z0-9_]*/).map((s) => Field.fromString(s));

// Parser for custom fields (customField function)
const customField = fn.map((functionValue) => {
  const { value: fnName, args } = functionValue;

  // Check if function name is "customfield" (case-insensitive)
  if (fnName.toLowerCase() !== 'customfield') {
    throw new GlqlAnalyzeError.UnrecognizedFunction(fnName, args);
  }

  // Check if exactly one argument is provided
  if (args.length !== 1) {
    throw new GlqlAnalyzeError.UnexpectedNumberOfArguments(fnName, 1, args.length);
  }

  // Return the custom field with the first argument
  return new Field.Custom(args[0]!);
});

// Parser for all fields - this should consume only the field name
export const field = alt(
  customField,
  staticField,
  all.map((input) => {
    throw new GlqlParseError(GlqlParseErrorType.MissingField, input);
  }),
);

// Parser for operators
const operator = alt(
  regexp(/in/i).map(() => Operator.In),
  string('=').map(() => Operator.Equal),
  string('!=').map(() => Operator.NotEqual),
  string('>=').map(() => Operator.GreaterThanEquals),
  string('>').map(() => Operator.GreaterThan),
  string('<=').map(() => Operator.LessThanEquals),
  string('<').map(() => Operator.LessThan),
  all.map((input) => {
    throw new GlqlParseError(GlqlParseErrorType.MissingOperator, input);
  }),
);

// Parser for expressions
export const expression = seqMap(
  field,
  optWhitespace,
  operator,
  optWhitespace,
  value,
  (field, _, operator, __, val) => new Expression(field as Field, operator, val),
);
