import { field, expression } from './expression';
import { Field } from '../types/field';
import { Operator } from '../types/operator';
import { Expression } from '../types/expression';
import { Value } from '../types/value';

describe('Expression Parser', () => {
  describe('field', () => {
    it('should parse label field', () => {
      const result = field.parse('label');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Field);
        expect(result.value.toString()).toBe('label');
      }
    });

    it('should parse unknown field', () => {
      const result = field.parse('thing');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Field.Unknown);
        expect(result.value.name).toBe('unknownField');
      }
    });

    it('should parse another unknown field', () => {
      const result = field.parse('here');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Field.Unknown);
        expect(result.value.name).toBe('unknownField');
      }
    });

    it('should fail on numeric input', () => {
      expect(() => field.parse('123')).toThrow();
    });

    it('should fail on empty input', () => {
      expect(() => field.parse('')).toThrow();
    });
  });

  describe('custom_field', () => {
    it('should parse customField with single argument', () => {
      const result = field.parse('customField("Subscription")');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Field.Custom);
        expect(result.value.name).toBe('customField');
      }
    });

    it('should throw on customField with no arguments', () => {
      expect(() => {
        field.parse('customField()');
      }).toThrow();
    });

    it('should throw on customField with multiple arguments', () => {
      expect(() => {
        field.parse('customField("Subscription", "Free")');
      }).toThrow();
    });

    it('should throw on customField with three arguments', () => {
      expect(() => {
        field.parse('customField("Subscription", "Free", "Bar")');
      }).toThrow();
    });
  });

  describe('operator', () => {
    it('should parse IN operator', () => {
      const result = expression.parse("label IN ('rest')");
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value.operator).toBe(Operator.In);
      }
    });

    it('should parse = operator', () => {
      const result = expression.parse("label = 'rest'");
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value.operator).toBe(Operator.Equal);
      }
    });

    it('should parse != operator', () => {
      const result = expression.parse("label != 'rest'");
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value.operator).toBe(Operator.NotEqual);
      }
    });

    it('should parse > operator', () => {
      const result = expression.parse("label > 'rest'");
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value.operator).toBe(Operator.GreaterThan);
      }
    });

    it('should parse < operator', () => {
      const result = expression.parse("label < 'rest'");
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value.operator).toBe(Operator.LessThan);
      }
    });

    it('should parse >= operator', () => {
      const result = expression.parse("label >= 'rest'");
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value.operator).toBe(Operator.GreaterThanEquals);
      }
    });

    it('should parse <= operator', () => {
      const result = expression.parse("label <= 'rest'");
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value.operator).toBe(Operator.LessThanEquals);
      }
    });

    it('should fail on invalid operator', () => {
      expect(() => expression.parse('label invalid rest')).toThrow();
    });

    it('should fail on numeric operator', () => {
      expect(() => expression.parse('label 123 rest')).toThrow();
    });

    it('should fail on empty operator', () => {
      expect(() => expression.parse('label')).toThrow();
    });
  });

  describe('expression', () => {
    it('should parse complete expression', () => {
      const result = expression.parse('label = "backend"');
      expect(result.status).toBe(true);
      if (result.status) {
        expect(result.value).toBeInstanceOf(Expression);
        expect(result.value.field.toString()).toBe('label');
        expect(result.value.operator).toBe(Operator.Equal);
        expect(result.value.value).toBeInstanceOf(Value.Quoted);
        if (result.value.value instanceof Value.Quoted) {
          expect(result.value.value.value).toBe('backend');
        }
      }
    });

    it('should fail on incomplete expression (missing operator and value)', () => {
      expect(() => expression.parse('label')).toThrow();
    });
  });
});
