import { seqMap, string, regexp, optWhitespace, sepBy, seq, alt } from 'parsimmon';
import { Dimension } from '../types/dimension';
import { DimensionFunction } from '../types/dimension_function';
import { TimeSegmentType } from '../utils/date_time/time_segment_type';
import { TimeUnit } from '../utils/date_time/time_unit';
import { parseFields } from './fields';
import { username } from './special';
import { str } from './literals';
import { Value } from '../types/value';
import { ReferenceType } from '../types/reference_type';

// Parser for time dimension segments (e.g., timeSegment(1d))
const timeDimensionSegment = seqMap(
  optWhitespace,
  regexp(/timesegment/i),
  optWhitespace,
  string('('),
  optWhitespace,
  regexp(/[+-]?\d+[dwmy]/),
  optWhitespace,
  string(')'),
  (_, __, ___, ____, _____, relativeDateStr) => {
    // Parse the relative date string (e.g., "1d", "2m", "3y")
    const match = relativeDateStr.match(/^([+-]?)(\d+)([dwmy])$/);
    if (!match) {
      throw new Error(`Invalid relative date format: ${relativeDateStr}`);
    }

    const [, sign, number, unit] = match;
    if (!number || !unit) {
      throw new Error(`Invalid relative date format: ${relativeDateStr}`);
    }
    const quantity = parseInt(number, 10);
    const signValue = sign === '-' ? -1 : 1;

    let timeUnit: TimeUnit;
    switch (unit) {
      case 'd':
        timeUnit = TimeUnit.Day;
        break;
      case 'w':
        timeUnit = TimeUnit.Week;
        break;
      case 'm':
        timeUnit = TimeUnit.Month;
        break;
      case 'y':
        timeUnit = TimeUnit.Year;
        break;
      default:
        throw new Error(`Invalid time unit: ${unit}`);
    }

    return new DimensionFunction.Time(
      signValue * quantity,
      timeUnit,
      TimeSegmentType.FromStartOfUnit,
    );
  },
);

const userDimensionSegment = seqMap(
  seq(optWhitespace, regexp(/users/i), optWhitespace, string('('), optWhitespace),
  sepBy(
    alt(
      username,
      str.map((s) => new Value.Reference(ReferenceType.User, s.value)),
    ),
    seq(optWhitespace, string(','), optWhitespace),
  ),
  seq(optWhitespace, string(')')),
  (_, users) => {
    return new DimensionFunction.User(users);
  },
);

const dimensionSegment = alt(timeDimensionSegment, userDimensionSegment);

// Parser for individual dimensions (e.g., timeSegment(1d) on created)
const dimension = seqMap(
  dimensionSegment,
  optWhitespace,
  regexp(/on/i),
  optWhitespace,
  regexp(/[^,]+/), // Capture everything until comma or end
  (segment, _, __, ___, fieldStr) => {
    // Parse the field string and convert it to a DisplayField
    const fields = parseFields(fieldStr);
    if (fields.length === 0) {
      throw new Error("No field found after 'on'");
    }
    const field = fields[0];
    if (!field) {
      throw new Error("No field found after 'on'");
    }
    return new Dimension(segment, field);
  },
);

// Parser for dimension lists (comma-separated)
const dimensionList = sepBy(dimension, seq(optWhitespace, string(','), optWhitespace))
  .assert((dimensions) => dimensions.length > 0, 'At least one dimension is required')
  .assert((dimensions) => dimensions.length <= 2, 'Only up to two dimensions are supported');

// Main parser for dimensions
export const parseDimensions = (input: string): Dimension[] => {
  const result = dimensionList.parse(input.trim());
  if (result.status) {
    return result.value;
  }
  throw new Error(`Failed to parse dimensions: ${result.expected.join(', ')}`);
};
