import { parseDimensions } from './dimensions';
import { DimensionFunction } from '../types/dimension_function';
import { DisplayField } from '../types/display_field';
import { TimeUnit } from '../utils/date_time/time_unit';
import { TimeSegmentType } from '../utils/date_time/time_segment_type';

describe('Dimensions Parser', () => {
  describe('valid_time_segment_dimension', () => {
    it('should parse timeSegment(1d) on created', () => {
      const result = parseDimensions('timeSegment(1d) on created');
      expect(result).toHaveLength(1);

      const dimension = result[0]!;
      expect(dimension.fn).toBeInstanceOf(DimensionFunction.Time);
      expect(dimension.field).toBeInstanceOf(DisplayField.Static);

      const timeSegment = dimension.fn as DimensionFunction.Time;
      expect(timeSegment.quantity).toBe(1);
      expect(timeSegment.unit).toBe(TimeUnit.Day);
      expect(timeSegment.timeSegmentType).toBe(TimeSegmentType.FromStartOfUnit);

      const field = dimension.field as DisplayField.Static;
      expect(field.field.displayFieldName).toBe('createdAt');
    });

    it('should parse timeSegment(1d) on createdAt', () => {
      const result = parseDimensions('timeSegment(1d) on createdAt');
      expect(result).toHaveLength(1);

      const dimension = result[0]!;
      expect(dimension.fn).toBeInstanceOf(DimensionFunction.Time);
      expect(dimension.field).toBeInstanceOf(DisplayField.Static);

      const timeSegment = dimension.fn as DimensionFunction.Time;
      expect(timeSegment.quantity).toBe(1);
      expect(timeSegment.unit).toBe(TimeUnit.Day);
      expect(timeSegment.timeSegmentType).toBe(TimeSegmentType.FromStartOfUnit);

      const field = dimension.field as DisplayField.Static;
      expect(field.field.displayFieldName).toBe('createdAt');
    });

    it('should parse timeSegment(1m) on merged', () => {
      const result = parseDimensions('timeSegment(1m) on merged');
      expect(result).toHaveLength(1);

      const dimension = result[0]!;
      expect(dimension.fn).toBeInstanceOf(DimensionFunction.Time);
      expect(dimension.field).toBeInstanceOf(DisplayField.Static);

      const timeSegment = dimension.fn as DimensionFunction.Time;
      expect(timeSegment.quantity).toBe(1);
      expect(timeSegment.unit).toBe(TimeUnit.Month);
      expect(timeSegment.timeSegmentType).toBe(TimeSegmentType.FromStartOfUnit);

      const field = dimension.field as DisplayField.Static;
      expect(field.field.displayFieldName).toBe('mergedAt');
    });

    it('should parse timeSegment(1y) on closed', () => {
      const result = parseDimensions('timeSegment(1y) on closed');
      expect(result).toHaveLength(1);

      const dimension = result[0]!;
      expect(dimension.fn).toBeInstanceOf(DimensionFunction.Time);
      expect(dimension.field).toBeInstanceOf(DisplayField.Static);

      const timeSegment = dimension.fn as DimensionFunction.Time;
      expect(timeSegment.quantity).toBe(1);
      expect(timeSegment.unit).toBe(TimeUnit.Year);
      expect(timeSegment.timeSegmentType).toBe(TimeSegmentType.FromStartOfUnit);

      const field = dimension.field as DisplayField.Static;
      expect(field.field.displayFieldName).toBe('closedAt');
    });

    it("should parse timeSegment(1d) on createdAt as 'Date created'", () => {
      const result = parseDimensions("timeSegment(1d) on createdAt as 'Date created'");
      expect(result).toHaveLength(1);

      const dimension = result[0]!;
      expect(dimension.fn).toBeInstanceOf(DimensionFunction.Time);
      expect(dimension.field).toBeInstanceOf(DisplayField.Aliased);

      const timeSegment = dimension.fn as DimensionFunction.Time;
      expect(timeSegment.quantity).toBe(1);
      expect(timeSegment.unit).toBe(TimeUnit.Day);
      expect(timeSegment.timeSegmentType).toBe(TimeSegmentType.FromStartOfUnit);

      const aliasedField = dimension.field as DisplayField.Aliased;
      expect(aliasedField.key).toBe('createdAt');
      expect(aliasedField.field).toBeInstanceOf(DisplayField.Static);

      const staticField = aliasedField.field as DisplayField.Static;
      expect(staticField.field.displayFieldName).toBe('createdAt');
    });
  });

  describe('invalid_dimensions', () => {
    it("should fail on incomplete dimension (missing 'on' clause)", () => {
      expect(() => parseDimensions('timeSegment(1d)')).toThrow();
    });

    it('should fail on more than 2 dimensions', () => {
      expect(() =>
        parseDimensions(
          'timeSegment(1d) on closed, timeSegment(1d) on created, users(@user1,@user2) on assignee',
        ),
      ).toThrow();
    });
  });

  describe('edge_cases', () => {
    it('should parse negative time values', () => {
      const result = parseDimensions('timeSegment(-2w) on created');
      expect(result).toHaveLength(1);

      const dimension = result[0]!;
      const timeSegment = dimension.fn as DimensionFunction.Time;
      expect(timeSegment.quantity).toBe(-2);
      expect(timeSegment.unit).toBe(TimeUnit.Week);
    });

    it('should parse case insensitive timeSegment', () => {
      const result = parseDimensions('TIMESEGMENT(1d) on created');
      expect(result).toHaveLength(1);

      const dimension = result[0]!;
      expect(dimension.fn).toBeInstanceOf(DimensionFunction.Time);
    });

    it("should parse case insensitive 'on' keyword", () => {
      const result = parseDimensions('timeSegment(1d) ON created');
      expect(result).toHaveLength(1);

      const dimension = result[0]!;
      expect(dimension.field).toBeInstanceOf(DisplayField.Static);
    });

    it('should handle whitespace variations', () => {
      const result = parseDimensions('  timeSegment(1d)  on  created  ');
      expect(result).toHaveLength(1);

      const dimension = result[0]!;
      expect(dimension.fn).toBeInstanceOf(DimensionFunction.Time);
      expect(dimension.field).toBeInstanceOf(DisplayField.Static);
    });
  });
});
