import { parseExpressions } from './parser';
import { parseFields } from './parser/fields';
import { transform as transformData } from './transformer/data';
import { CompileContext } from './types/compile_context';
import { CompileOutput } from './types/compile_output';
import { Context } from './types/context';
import { DisplayField } from './types/display_field';
import { Sort } from './types/sort';
import { TransformContext } from './types/transform_context';
import { featureFlags } from './utils/feature_flags';
import { TransformOutput } from './types/transform_output';
import { parseDimensions } from './parser/dimensions';
import { Dimension } from './types/dimension';
import { Query } from './types/query';

export function compile(query: string, ctx: CompileContext, isSubquery: boolean = false) {
  try {
    if (typeof ctx.featureFlags?.glqlWorkItems === 'boolean') {
      featureFlags.glqlWorkItems = ctx.featureFlags?.glqlWorkItems;
    }

    let fields: DisplayField[] = [];
    let groupBy: Dimension[] = [];
    let aggregate: DisplayField[] = [];
    if (ctx.groupBy && ctx.aggregate) {
      groupBy = parseDimensions(ctx.groupBy);
      aggregate = parseFields(ctx.aggregate);
    } else if (ctx.fields) {
      fields = parseFields(ctx.fields);
    }

    return Query.compile({
      expressions: parseExpressions(query),
      fields,
      groupBy,
      aggregate,
      sort: ctx.sort ? Sort.fromString(ctx.sort) : undefined,
      isSubquery,
      context: new Context({
        username: ctx.username,
        project: ctx.project,
        group: ctx.group,
      }),
    }).toGraphQL();
  } catch (error) {
    return CompileOutput.error((error as Error).message);
  }
}

export function transform(data: unknown, ctx: TransformContext) {
  try {
    let fields: DisplayField[] = [];
    let dimensions: Dimension[] = [];
    let metrics: DisplayField[] = [];
    if (ctx.groupBy && ctx.aggregate) {
      dimensions = parseDimensions(ctx.groupBy);
      metrics = parseFields(ctx.aggregate);

      fields = dimensions.map((d) => d.field);
      fields = fields.concat(metrics);
    } else if (ctx.fields) {
      fields = parseFields(ctx.fields);
    } else {
      throw new Error('No fields provided');
    }

    return TransformOutput.success(
      transformData(data, fields as [DisplayField, ...DisplayField[]]),
      fields,
    );
  } catch (error) {
    return TransformOutput.error((error as Error).message);
  }
}
