import { FieldMapping } from './types/index';
import { FieldType } from './types/field_type';
import { Value } from './types/value';
import { FieldName } from './types/field';
import { SourceName } from './types/source';
import { Field } from './types/field';
import { Context } from './types/context';
import { featureFlags } from './utils/feature_flags';
import { ReferenceType } from './types/reference_type';
import { RelationshipType } from './types/relationship_type';
import { parseWorkItemReference, uniqueId } from './utils/common';
import { Variable } from './types/variable';
import { parseEpicReference } from './utils/common';
import { compile } from './lib';
import { CompileContext } from './types/compile_context';

/// Attribute function for work items epic mapping (parent work items)
function parentFilterKey(fieldType: FieldType, value: Value) {
  if (fieldType.isNullable && (value instanceof Value.Null || value instanceof Value.Token)) {
    return 'parentWildcardId';
  }
  return 'parentIds';
}

/// Attribute function for legacy epic mapping
function epicFilterKey(fieldType: FieldType, value: Value) {
  if (fieldType.isNullable && (value instanceof Value.Null || value instanceof Value.Token)) {
    return 'epicWildcardId';
  }
  return 'epicId';
}

export function getFieldMapping(field: Field, context: Context): FieldMapping | void {
  field = field.dealias();

  // Epic field with work items API (epics are treated as parent work items)
  if (
    field.name === FieldName.Epic &&
    featureFlags.glqlWorkItems &&
    context.source.name === SourceName.Issues
  ) {
    return {
      fieldType: new FieldType.StringLike()
        .or(new FieldType.NumberLike())
        .or(new FieldType.ReferenceLike(ReferenceType.Epic))
        .or(
          new FieldType.ListLike(
            RelationshipType.HasOne,
            new FieldType.StringLike()
              .or(new FieldType.NumberLike())
              .or(new FieldType.ReferenceLike(ReferenceType.Epic)),
          ),
        ),
      variableType: 'WorkItemID!',
      getAttribute: parentFilterKey,
    };
  }

  // Epic field with legacy API
  if (field.name === FieldName.Epic && !featureFlags.glqlWorkItems) {
    return {
      fieldType: new FieldType.StringLike()
        .or(new FieldType.NumberLike())
        .or(new FieldType.ReferenceLike(ReferenceType.Epic))
        .or(new FieldType.Nullable()),
      variableType: 'String',
      getAttribute: epicFilterKey,
    };
  }

  // Parent field with work items API
  if (field.name === FieldName.Parent && featureFlags.glqlWorkItems) {
    return {
      fieldType: new FieldType.StringLike()
        .or(new FieldType.NumberLike())
        .or(new FieldType.ReferenceLike(ReferenceType.WorkItem))
        .or(new FieldType.ReferenceLike(ReferenceType.Epic))
        .or(
          new FieldType.ListLike(
            RelationshipType.HasOne,
            new FieldType.StringLike()
              .or(new FieldType.NumberLike())
              .or(new FieldType.ReferenceLike(ReferenceType.WorkItem))
              .or(new FieldType.ReferenceLike(ReferenceType.Epic)),
          ),
        ),
      variableType: 'WorkItemID!',
      getAttribute: parentFilterKey,
    };
  }
}

/// Common epic subquery transformation logic with configurable variable type
export function transformEpicSubquery(
  f: Field,
  v: Value,
  context: Context,
  variableType: string,
): Value {
  if (v instanceof Value.List) {
    return new Value.List(v.value.map((v) => transformEpicSubquery(f, v, context, variableType)));
  }

  let id: string;
  if (v instanceof Value.Number) id = v.value.toString();
  else if (v instanceof Value.Quoted) id = v.value;
  else if (v instanceof Value.Reference) {
    id = v.value;
    if (v.referenceType === ReferenceType.Epic && !id.includes('&')) id = `&${id}`;
    if (v.referenceType === ReferenceType.WorkItem && !id.includes('#')) id = `#${id}`;
  } else return v;

  const parentGroup = context.group
    ? context.group
    : (context.project || '').split('/').slice(0, -1).join('/');

  const subqueryContext: CompileContext = {
    fields: 'id',
    group: context.group,
    project: context.project,
  };

  const isEpic = f.name === FieldName.Epic;
  const variableKey = uniqueId(isEpic ? 'epicId' : 'parentId');

  const subquery = (query: string) => {
    context.variables.push(
      new Variable(variableKey, variableType, compile(query, subqueryContext, true).output),
    );

    return new Value.Subquery(query, new Variable(variableKey, variableType), subqueryContext);
  };

  // Parse the epic reference using the utility function
  if (isEpic) {
    const { groupPath, id: epicId } = parseEpicReference(id, parentGroup);
    return subquery(
      `type = Epic AND id = ${epicId} AND includeSubgroups = false AND group = "${groupPath}"`,
    );
  } else {
    const { projectPath, groupPath, id: workItemId } = parseWorkItemReference(id, parentGroup);
    let query = `id = ${workItemId}`;
    if (projectPath) {
      query += ` AND project = "${projectPath}"`;
    } else if (groupPath) {
      query += ` AND type = Epic AND group = "${groupPath}" AND includeSubgroups = false`;
    } else {
      query += ` AND group = "${parentGroup}"`;
    }
    return subquery(query);
  }
}
