import { Field, FieldName } from './types/field';
import { FieldType } from './types/field_type';
import { Operator } from './types/operator';
import { Source } from './types/source';
import { Value } from './types/value';

export const enum GlqlParseErrorType {
  UnknownError,
  InvalidRelativeDateUnit,
  UnterminatedString,
  MissingField,
  MissingOperator,
  MissingValue,
  InvalidUsername,
  InvalidMilestone,
  InvalidLabel,
  InvalidIteration,
  InvalidEpic,
  UnexpectedToken,
}

export enum InvalidScopeType {
  Project,
  Group,
  Namespace,
}

export abstract class GlqlError extends Error {}

export class GlqlParseError extends GlqlError {
  constructor(
    public code: GlqlParseErrorType,
    public input: string,
  ) {
    let message;
    switch (code) {
      case GlqlParseErrorType.InvalidRelativeDateUnit:
        message = `Unexpected character \`${input}\`. Expected d (day), w (week), m (month), or y (year)`;
        break;
      case GlqlParseErrorType.UnterminatedString:
        message = `Unterminated string \`${input}\``;
        break;
      case GlqlParseErrorType.MissingField:
        message = input
          ? `Expected valid field near \`${input}\``
          : `Unexpected end of input, expected valid field`;
        break;
      case GlqlParseErrorType.MissingOperator:
        message = input
          ? `Expected valid operator near \`${input}\``
          : `Unexpected end of input, expected valid operator`;
        break;
      case GlqlParseErrorType.MissingValue:
        message = input
          ? `Expected valid value near \`${input}\``
          : `Unexpected end of input, expected valid value`;
        break;
      case GlqlParseErrorType.InvalidUsername:
        message = `Invalid username reference \`${input}\``;
        break;
      case GlqlParseErrorType.InvalidMilestone:
        message = `Invalid milestone reference \`${input}\``;
        break;
      case GlqlParseErrorType.InvalidLabel:
        message = `Invalid label reference \`${input}\``;
        break;
      case GlqlParseErrorType.InvalidIteration:
        message = `Invalid iteration reference \`${input}\``;
        break;
      case GlqlParseErrorType.InvalidEpic:
        message = `Invalid epic reference \`${input}\``;
        break;
      case GlqlParseErrorType.UnexpectedToken:
        message = `Unexpected token near \`${input}\``;
        break;
      case GlqlParseErrorType.UnknownError:
      default:
        message = `Unknown error near \`${input}\``;
        break;
    }

    super(message);
  }
}

export abstract class GlqlAnalyzeError extends Error {}

export namespace GlqlAnalyzeError {
  export class ExpectedFieldToBeArray extends GlqlAnalyzeError {
    constructor(public value: string) {
      super(`Expected '${value}' to be an array`);
    }
  }

  export class InvalidDateFormat extends GlqlAnalyzeError {
    constructor(public date: string) {
      super(`Invalid date format \`${date}\`. Expected format: yyyy-mm-dd`);
    }
  }

  export class UnrecognizedFunction extends GlqlAnalyzeError {
    constructor(
      public fnName: string,
      public args: string[],
    ) {
      super(`Unrecognized function: ${fnName}(${args.map((a) => `"${a}"`).join(', ')})`);
    }
  }

  export class UnexpectedNumberOfArguments extends GlqlAnalyzeError {
    constructor(
      public fnName: string,
      public expected: number,
      public actual: number,
    ) {
      super(
        `Unexpected number of arguments for function: \`${fnName}\`. Expected: ${expected}, Actual: ${actual}`,
      );
    }
  }

  export class UnrecognizedField extends GlqlAnalyzeError {
    constructor(public field: Field) {
      super(`\`${field}\` is not a recognized field.`);
    }
  }

  export class SourceNotProvided extends GlqlAnalyzeError {
    constructor() {
      super('Source is required');
    }
  }

  export class UnrecognizedFieldForSource extends GlqlAnalyzeError {
    constructor(
      public field: Field,
      public source: Source,
    ) {
      super(`\`${field}\` is not a recognized field for ${source}.`);
    }
  }

  export class UnrecognizedSortFieldForSource extends GlqlAnalyzeError {
    constructor(
      public field: Field,
      public source: Source,
      public supportedFields: Field[],
    ) {
      super(
        `\`${field}\` is not a recognized sort field for ${source}. Supported fields to sort by: ${supportedFields
          .map((f) => `\`${f}\``)
          .join(', ')}.`,
      );
    }
  }

  export class InvalidSortOrder extends GlqlAnalyzeError {
    constructor(public order: string) {
      super(`Invalid sort order: \`${order}\`. Valid sort orders: \`asc\`, \`desc\`.`);
    }
  }

  export class DuplicateOperatorForField extends GlqlAnalyzeError {
    constructor(
      public operator: Operator,
      public field: Field,
    ) {
      super(`Operator ${operator} can only be used once with \`${field}\`.`);
    }
  }

  export class UnsupportedOperatorForField extends GlqlAnalyzeError {
    constructor(
      public field: Field,
      public operator: Operator,
      public supportedOperators: Operator[],
    ) {
      super(
        `\`${field}\` does not support the ${operator} operator. Supported operators: ${supportedOperators.join(
          ', ',
        )}.`,
      );
    }
  }

  export class UnsupportedOperatorValueForField extends GlqlAnalyzeError {
    constructor(
      public field: Field,
      public operator: Operator,
      public value: Value,
      public supportedOperators: Operator[],
    ) {
      const operatorsString =
        supportedOperators.length > 0
          ? ` Supported operators: ${supportedOperators.join(', ')}.`
          : '';
      super(
        `\`${field}\` does not support the ${operator} operator for \`${value}\`.${operatorsString}`,
      );
    }
  }

  export class UnsupportedValueForField extends GlqlAnalyzeError {
    constructor(
      public field: Field,
      public value: Value,
      public supportedValueTypes: FieldType[],
    ) {
      let message = `\`${field}\` cannot be compared with \`${value}\`. Supported value types: ${supportedValueTypes.join(
        ', ',
      )}.`;
      field = field.dealias();

      // Epic and MergeRequest types cannot be combined with other types
      for (const token of ['Epic', 'MergeRequest']) {
        if (
          field.name === FieldName.Type &&
          value instanceof Value.List &&
          value.value.some((v) => v.eq(new Value.Token(token)))
        ) {
          const tokenValue = value.value.find((v) => v.eq(new Value.Token(token)))!;
          message = `Type \`${tokenValue}\` cannot be combined with other types (${value.value
            .filter((v) => !v.eq(new Value.Token(token)))
            .map((v) => `\`${v}\``)
            .join(', ')}) in the same query. Try using \`type = ${tokenValue}\` instead.`;
        }
      }

      if (
        (field.name === FieldName.Created || field.name === FieldName.Closed) &&
        value instanceof Value.Bool
      ) {
        const isCreated = field.name === FieldName.Created;
        const state = isCreated === value.value ? 'opened' : 'closed';
        message += ` Did you mean \`state = ${state}\`?`;
      }

      super(message);
    }
  }

  export class IncorrectFieldPairing extends GlqlAnalyzeError {
    constructor(
      public field: Field,
      public supportedFields: Field[],
    ) {
      super(
        `\`${field}\` can only be used with: ${supportedFields.map((f) => `\`${f}\``).join(', ')}.`,
      );
    }
  }

  export class InvalidScope extends GlqlAnalyzeError {
    constructor(public scopeType: InvalidScopeType) {
      let message;
      switch (scopeType) {
        case InvalidScopeType.Project:
          message = 'Project does not exist or you do not have access to it';
          break;
        case InvalidScopeType.Group:
          message = 'Group does not exist or you do not have access to it';
          break;
        case InvalidScopeType.Namespace:
          message = 'Namespace does not exist or you do not have access to it';
          break;
      }

      super(message);
    }
  }

  export class ArrayTooLarge extends GlqlAnalyzeError {
    constructor(
      public field: Field,
      public maxAllowed: number,
    ) {
      super(`\`${field}\` field exceeds maximum limit of ${maxAllowed} items.`);
    }
  }

  export class UnsupportedDimensionFunction extends GlqlAnalyzeError {
    constructor(public fnName: string) {
      super(`\`${fnName}\` is not a supported dimension function.`);
    }
  }

  export class MaxDimensionsExceeded extends GlqlAnalyzeError {
    constructor() {
      super('Only one dimension is supported');
    }
  }

  export class CannotCombineValues extends GlqlAnalyzeError {
    constructor(
      public a: Value,
      public b: Value,
    ) {
      super(`Cannot combine values \`${a}\` and \`${b}\``);
    }
  }

  export class UnknownError extends GlqlAnalyzeError {
    constructor() {
      super(`Unknown error occurred`);
    }
  }
}
