import { Query } from '../types/query';
import { Context } from '../types/context';
import { Value, ValueType } from '../types/value';
import { Expression } from '../types/expression';
import { Field, FieldName } from '../types/field';
import { Sort } from '../types/sort';
import { OperatorType } from '../types/operator';
import { SourceName } from '../types/source';
import { featureFlags } from '../utils/feature_flags';
import { Variable } from '../types/variable';
import { getFieldMapping, transformEpicSubquery } from '../field_mapping';

export class GraphQLFilters {
  and: Record<string, Value> = {};
  or: Record<string, Value> = {};
  not: Record<string, Value> = {};
  constraints: Record<string, Value> = {};
  customFields: Record<string, Value> = {};

  constructor(query?: Query) {
    if (!query) return;

    for (const expr of query.expressions) {
      const key = toGraphQLFilterKey(expr, query.context);
      const value = toGraphQLFilterValue(expr, query.context);
      const field = expr.field.dealias();

      if (field instanceof Field.Custom) {
        this.customFields[field.customName] = value;
        continue;
      }

      switch (expr.operator.type) {
        case OperatorType.NotEqual:
          this.not[key] = value;
          break;
        case OperatorType.In:
          this.or[key] = value;
          break;
        default:
          this.and[key] = value;
          break;
      }
    }

    this.constraints['before'] = new Value.Token('$before');
    this.constraints['after'] = new Value.Token('$after');
    this.constraints['first'] = new Value.Token('$limit');
    query.context.variables.push(new Variable('before', 'String'));
    query.context.variables.push(new Variable('after', 'String'));
    query.context.variables.push(new Variable('limit', 'Int'));

    if (query.sort) {
      this.constraints['sort'] = new Value.Token(toGraphQLSort(query.sort));
    }

    if (query.context.group && !query.containsField(new Field(FieldName.IncludeSubgroups))) {
      let fieldName = 'includeSubgroups';

      if (featureFlags.glqlWorkItems && query.context.source.name === SourceName.Issues)
        fieldName = 'includeDescendants';

      if (!featureFlags.glqlWorkItems && query.context.source.name === SourceName.Epics)
        fieldName = 'includeDescendantGroups';

      this.constraints[fieldName] = new Value.Bool(true);
    }

    if (
      query.context.group &&
      query.containsFieldWithValue(new Field(FieldName.Type), new Value.Token('EPIC'))
    ) {
      this.constraints['excludeProjects'] = new Value.Bool(true);
    }

    if (
      query.containsField(new Field(FieldName.Epic)) ||
      query.containsField(new Field(FieldName.Parent))
    ) {
      this.constraints[
        featureFlags.glqlWorkItems ? 'includeDescendantWorkItems' : 'includeSubepics'
      ] = new Value.Bool(true);
    }
  }

  clone(): GraphQLFilters {
    const clone = new GraphQLFilters();
    clone.and = { ...this.and };
    clone.or = { ...this.or };
    clone.not = { ...this.not };
    clone.constraints = { ...this.constraints };
    clone.customFields = { ...this.customFields };
    return clone;
  }

  toString(): string {
    let filters = '';
    const formatValue = (value: Value): string => {
      switch (value.type) {
        case ValueType.Null:
          return 'null';
        case ValueType.Bool:
          return value.value ? 'true' : 'false';
        case ValueType.Number:
          return (value as Value.Number).value.toString();
        case ValueType.Token:
          return (value as Value.Token).value;
        case ValueType.Quoted:
        case ValueType.Reference:
        case ValueType.Date:
          return `"${value.value}"`;
        case ValueType.List:
          return `[${(value as Value.List).value.map(formatValue).join(', ')}]`;
        case ValueType.RelativeDate:
        case ValueType.Function:
          throw new Error('Error: Cannot generate code before function evaluation.');
        case ValueType.Subquery:
          return `$${(value as Value.Subquery).variable.key}`;
        default:
          throw new Error(`Unknown value type: ${value.type}`);
      }
    };

    const formatFilters = (filters: Record<string, Value>, wrapper?: string): string => {
      if (Object.keys(filters).length === 0) return '';

      const joined = Object.entries(filters)
        .map(([key, value]) => `${key}: ${formatValue(value)}`)
        .join(' ');

      return wrapper ? `${wrapper}: { ${joined} }` : `${joined} `;
    };

    const formatCustomFieldFilters = (customFields: Record<string, Value>): string => {
      if (Object.keys(customFields).length === 0) return '';

      const joined = Object.entries(customFields)
        .map(
          ([name, value]) =>
            `{customFieldName: "${name}", selectedOptionValues: ${formatValue(value)}}`,
        )
        .join(' ');

      return `customField: [${joined}] `;
    };

    filters += formatFilters(this.and);
    filters += formatFilters(this.or, 'or');
    filters += formatFilters(this.not, 'not');
    filters += formatFilters(this.constraints);
    filters += formatCustomFieldFilters(this.customFields);

    return !filters ? filters : `(${filters})`;
  }
}

function toHealthStatus(value: Value): Value {
  if (value instanceof Value.Quoted) {
    const status = {
      'on track': 'onTrack',
      'needs attention': 'needsAttention',
      'at risk': 'atRisk',
    }[value.value.toLowerCase()];

    if (status) return new Value.Token(status);
  }

  return value;
}

function toIterationCadenceId(value: Value): Value {
  if (value instanceof Value.Number)
    return new Value.Quoted(`gid://gitlab/Iterations::Cadence/${value.value}`);

  if (value instanceof Value.List) return new Value.List(value.value.map(toIterationCadenceId));

  return value;
}

function toTypeParam(s: string): string {
  switch (s) {
    case 'TESTCASE':
      return 'TEST_CASE';
    case 'KEYRESULT':
      return 'KEY_RESULT';
    case 'MERGEREQUEST':
      return 'MERGE_REQUEST';
    default:
      return s;
  }
}

function toGraphQLFilterValue(expr: Expression, context: Context): Value {
  const field = expr.field.dealias();
  const value = expr.value;

  if (field.name === FieldName.Id && value instanceof Value.Number)
    return new Value.Quoted(value.value.toString());

  if (field.name === FieldName.Id && value instanceof Value.List)
    return new Value.List(value.value.map((i) => new Value.Quoted(i.toString())));

  if (field.name === FieldName.Assignee && value instanceof Value.Number)
    return new Value.Quoted(value.value.toString());

  if (
    (field.name === FieldName.Label || field.name === FieldName.Approver) &&
    value instanceof Value.Token
  )
    return new Value.Quoted(value.toString());

  if (
    (field.name === FieldName.Label ||
      field.name === FieldName.Assignee ||
      field.name === FieldName.Approver) &&
    value instanceof Value.Null
  )
    return new Value.Quoted('NONE');

  if (
    (field.name === FieldName.Weight || field.name === FieldName.Epic) &&
    value instanceof Value.Null
  )
    return new Value.Token('NONE');

  if (field.name === FieldName.Weight && value instanceof Value.Number)
    return new Value.Quoted(value.value.toString());

  if (field.name === FieldName.State && value instanceof Value.Token)
    return new Value.Token(value.toString().toLowerCase());

  if (field.name === FieldName.Status && value instanceof Value.Quoted)
    return new Value.Token(`{name: "${value.value}"}`);

  if (field.name === FieldName.Type && value instanceof Value.Token)
    return new Value.Token(toTypeParam(value.toString()));

  if (field.name === FieldName.Type && value instanceof Value.List)
    return new Value.List(value.value.map((i) => new Value.Token(toTypeParam(i.toString()))));

  if (field.name === FieldName.Health) return toHealthStatus(value);

  if (field.name === FieldName.Cadence) return toIterationCadenceId(value);

  if (field.name === FieldName.Subscribed && value instanceof Value.Bool)
    return new Value.Token(value.value ? 'EXPLICITLY_SUBSCRIBED' : 'EXPLICITLY_UNSUBSCRIBED');

  if (field.name === FieldName.Epic || field.name === FieldName.Parent) {
    const mapping = getFieldMapping(field, context);
    if (mapping) {
      const subquery = transformEpicSubquery(field, value, context, mapping.variableType);
      // graphql variables are not automatically coerced to lists,
      // so we need to wrap the subquery in a list, if it isn't already
      if (featureFlags.glqlWorkItems && subquery instanceof Value.Subquery) {
        return new Value.List([subquery]);
      }
      return subquery;
    }
  }

  return value;
}

function toGraphQLSort(sort: Sort): string {
  const field = sort.field.dealias();
  let sortFieldName = '';
  switch (field.name) {
    case FieldName.Merged:
      sortFieldName = 'merged_at';
      break;
    case FieldName.Closed:
      sortFieldName = 'closed_at';
      break;
    case FieldName.Title:
      sortFieldName = 'title';
      break;
    case FieldName.Popularity:
      sortFieldName = 'popularity';
      break;
    case FieldName.Milestone:
      sortFieldName = 'milestone_due';
      break;
    case FieldName.Updated:
      sortFieldName = 'updated';
      break;
    case FieldName.Created:
      sortFieldName = 'created';
      break;

    // Epics
    case FieldName.Start:
      sortFieldName = 'start_date';
      break;
    case FieldName.Due:
      sortFieldName = 'due_date';
      break;

    // Issues
    case FieldName.Weight:
      sortFieldName = 'weight';
      break;
    case FieldName.Health:
      sortFieldName = 'health_status';
      break;

    // Default
    default:
      return toGraphQLSort(new Sort(field, sort.order));
  }

  return `${sortFieldName.toUpperCase()}_${sort.order.toUpperCase()}`;
}

function toGraphQLFilterKey(expr: Expression, context: Context): string {
  const field = expr.field.dealias();
  const { operator, value } = expr;

  if (
    field.name === FieldName.Assignee &&
    (value instanceof Value.Number || value instanceof Value.Null)
  )
    return 'assigneeId';

  if (field.name === FieldName.Assignee && value instanceof Value.Token)
    return 'assigneeWildcardId';

  if (field.name === FieldName.Assignee && value instanceof Value.List && value.value.length === 0)
    return 'assigneeWildcardId';

  if (
    field.name === FieldName.Assignee &&
    operator.type === OperatorType.Equal &&
    context.source.name === SourceName.MergeRequests
  )
    return 'assigneeUsername';

  if (field.name === FieldName.Assignee) return 'assigneeUsernames';

  if (field.name === FieldName.Author && operator.type === OperatorType.In)
    return 'authorUsernames';

  if (field.name === FieldName.Author) return 'authorUsername';

  if (field.name === FieldName.Id) return 'iids';

  if (field.name === FieldName.Type) return 'types';

  if (field.name === FieldName.Approver) return 'approvedBy';

  if (field.name === FieldName.Merger) return 'mergedBy';

  if (field.name === FieldName.Reviewer && value instanceof Value.Token)
    return 'reviewerWildcardId';

  if (field.name === FieldName.Reviewer) return 'reviewerUsername';

  if (field.name === FieldName.Environment) return 'environmentName';

  if (field.name === FieldName.Epic || field.name === FieldName.Parent) {
    const mapping = getFieldMapping(field, context);
    if (mapping) return mapping.getAttribute(mapping.fieldType, expr.value);
  }

  if (field.name === FieldName.Label && operator.type === OperatorType.In) return 'labelNames';

  if (field.name === FieldName.Label) return 'labelName';

  if (field.name === FieldName.Health) return 'healthStatusFilter';

  if (field.name === FieldName.IncludeSubgroups) {
    if (featureFlags.glqlWorkItems && context.source.name === SourceName.Issues)
      return 'includeDescendants';

    if (!featureFlags.glqlWorkItems && context.source.name === SourceName.Epics)
      return 'includeDescendantGroups';

    return 'includeSubgroups';
  }

  if (field.name === FieldName.Iteration && value instanceof Value.Token)
    return 'iterationWildcardId';

  if (field.name === FieldName.Iteration) return 'iterationId';

  if (field.name === FieldName.Cadence) return 'iterationCadenceId';

  if (field.name === FieldName.Milestone && value instanceof Value.Token)
    return 'milestoneWildcardId';

  if (field.name === FieldName.Milestone && value instanceof Value.List && value.value.length === 0)
    return 'milestoneWildcardId';

  if (field.name === FieldName.Milestone) return 'milestoneTitle';

  if (field.name === FieldName.SourceBranch) return 'sourceBranches';

  if (field.name === FieldName.TargetBranch) return 'targetBranches';

  if (field.name === FieldName.MyReaction) return 'myReactionEmoji';

  if (field.name === FieldName.Project) {
    throw new Error('project interpreted as token instead of captured by optimizer');
  }

  if (field.name === FieldName.Group) {
    throw new Error('group interpreted as token instead of captured by optimizer');
  }

  if (
    field.name === FieldName.Weight &&
    (value instanceof Value.Null || value instanceof Value.Token)
  )
    return 'weightWildcardId';

  if (
    operator.type === OperatorType.GreaterThan ||
    operator.type === OperatorType.GreaterThanEquals
  )
    return `${field}After`;

  if (operator.type === OperatorType.LessThan || operator.type === OperatorType.LessThanEquals)
    return `${field}Before`;

  if (field instanceof Field.Unknown) return field.customName;

  return field.toString();
}
